/* 
   Http request handler
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 1999-2004, Joe Orton <joe@manyfish.co.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavRequest.cpp 432 2008-10-27 05:44:14Z yone $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <ctype.h>
#include <string.h>
#include <onion/DavAuthManager.h>
#include <onion/DavMemBuffer.h>
#include <onion/DavRequest.h>
#include <onion/DavResponseBody.h>
#include <onion/DavServerPolicy.h>
#include <onion/DavSocket.h>
#include <onion/DavWorkSession.h>
#include <onion/DavXmlBuffer.h>
#include <onion/HandlerConn.h>
#include <onion/HandlerKeepAlive.h>
#include <onion/HandlerLocation.h>
#include <onion/HandlerProxyConn.h>
#include <onion/RequestBodyProvider.h>
#include <onion/ResponseHeaderHandler.h>
#include <onion/PBCSMMemBuffer.h>
#include <onion/PBCSMXml.h>

#define RETRY_RET(retry, code, acode) \
		((((code) == OISECLOSED || (code) == OISERESET || \
		(code) == OISECONNABORTED || (code) == OISETRUNC) \
		&& retry) ? OI_RETRY : (acode))


CDavRequest::CDavRequest():
  m_pSession(NULL)
{
  Reset();
}

CDavRequest::~CDavRequest(void)
{
  ClearHandlers();
}

CDavWorkSession*
CDavRequest::GetSession()
{
  return m_pSession;
}

OI_REQ_TYPE
CDavRequest::GetMethod()
{
  return m_enuType;
}

const char*
CDavRequest::GetMethodStr()
{
  return m_strMethod.c_str();
}

int 
CDavRequest::GetStatusCode()
{
  return m_HttpStatus.m_nCode;
}

int 
CDavRequest::GetStatusClass()
{
  return m_HttpStatus.m_nCodeClass;
}

void 
CDavRequest::SetPersistent(bool bValue)
{
  m_pSession->SetPersistent(bValue);
}

void 
CDavRequest::ClearHandlers()
{
  OI_HANDLERMAP::iterator it;
  for(it = m_mapRespHeaderHandlers.begin();
      it != m_mapRespHeaderHandlers.end();
      it++){
    delete it->second;
  }
  m_mapRespHeaderHandlers.clear();
}

bool 
CDavRequest::AddHandler(CResponseHeaderHandler* pHandler, const char* pszName)
{
  OI_ASSERT(pHandler);
	
  bool bExists = false;
  OI_STRING_A strName = (pszName) ? pszName : pHandler->GetHeaderName();
  TrimLeftA(strName);
  TrimRightA(strName);
  //OI_DEBUG("Adding Handler <%s>\n", strName.c_str());
  //convert name into lower case
  MakeLowerA(strName);
  OI_HANDLERMAP::iterator it = m_mapRespHeaderHandlers.find(strName);
  if(it != m_mapRespHeaderHandlers.end()){
    bExists = true;
    delete it->second;
  }
  m_mapRespHeaderHandlers[strName] = pHandler;
  return bExists;
}

bool 
CDavRequest::RemoveHandler(const char *pszName)
{
  OI_ASSERT(pszName);
	
  OI_STRING_A strName = pszName;
  MakeLowerA(strName);

  OI_HANDLERMAP::iterator it = m_mapRespHeaderHandlers.find(strName);
  if(it != m_mapRespHeaderHandlers.end()){
    delete it->second;
    m_mapRespHeaderHandlers.erase(it);
    return true;
  }
  return false;
}

void
CDavRequest::ClearRequestHeaders()
{
  m_mapReqHeaders.clear();
}

void 
CDavRequest::AddRequestHeader(const char *pszName, const char *pszValue)
{
  OI_ASSERT(pszName);
  OI_STRING_A strValue;
  if(pszValue)
    strValue = pszValue;
  //OI_DEBUG("Adding Header <%s = %s>\n", pszName, pszValue);
  m_mapReqHeaders[pszName].push_back(strValue);
}

void
CDavRequest::SetRequestHeader(const char* pszName, const char* pszValue)
{
  RemoveRequestHeader(pszName);
  AddRequestHeader(pszName, pszValue);
}

bool 
CDavRequest::RemoveRequestHeader(const char *pszName)
{
  OI_ASSERT(pszName);
  bool bExists = false;
  OI_HEADERMAP::iterator it = m_mapReqHeaders.find(pszName);
  if(it != m_mapReqHeaders.end()){
    bExists = true;
    m_mapReqHeaders.erase(it);
  }
  return bExists;
}

void 
CDavRequest::addFixedHeaders()
{
  OI_STRING_A strUserAgent = m_pSession->m_strUserAgent;

  if(strUserAgent.length() > 0)
    SetRequestHeader("User-Agent", strUserAgent.c_str());
  /*
    if(m_pSession->IsHttp11() || m_pSession->IsUsingProxy())
    AddRequestHeader("Connection", "TE");
    else
    {
  */
  //???Squid does not accept the above header
/*
  SetRequestHeader("Connection", "TE, Keep-Alive");
  SetRequestHeader("Keep-Alive", "");
*/

  bool bUseSSL   = m_pSession->IsUsingSSL();
  bool bUseProxy = m_pSession->IsUsingProxy();

  if (!bUseProxy || (bUseSSL && bUseProxy && m_enuType != T_REQ_CONNECT))
  {
	// no use proxy, or ssl proxy w/o CONNECT method
	SetRequestHeader(OI_RSPHDR_CONN, "TE, Keep-Alive");
//	SetRequestHeader(OI_RSPHDR_KEEPALIVE, "");
  }
  else
  {
	  // use proxy with CONNECT method
	 if (m_enuType == T_REQ_CONNECT)
	 {
		 SetRequestHeader(OI_REQHDR_PROXYCONN, OI_RSPHDR_KEEPALIVE);
	 }
	 else
	 {
		// use proxy w/o CONNECT method
		SetRequestHeader(OI_REQHDR_PROXYCONN, "TE, Keep-Alive");
//		SetRequestHeader(OI_RSPHDR_KEEPALIVE, "");
	 }
  }

#ifdef HAVE_LIBZ
  if(m_pSession->IsUsingCompression())
    SetRequestHeader("Accept-Encoding", "gzip");
#endif /*HAVE_LIBZ*/

  /*
    }*/
}

void 
CDavRequest::stripEOL(char* pszBuf, size_t* punLen)
{
  char* pCh = &pszBuf[*punLen-1];
  while (pCh >= pszBuf && (*pCh == '\r' || *pCh == '\n')){
    *pCh-- = '\0';
    (*punLen)--;
  }
}

OI_RESULT 
CDavRequest::Create(CDavWorkSession* pSession,
		    OI_REQ_TYPE enuType,
		    const char *pszMethod,
		    const X& strURI)
{
  if(!pSession || !pszMethod)
    return OIGEINVALIDARG;

  if(((X&)strURI).Length() <= 0)
    return OIGEINVALIDARG;

  if(m_pSession) {
    if(m_pSession->m_pActiveRequest == this)
      return OIGEINVALIDSTATE;
    else
      Reset();
  }
  m_pSession = pSession;
  m_enuType = enuType;
  m_strMethod = pszMethod;
  m_strURI = strURI;
  addFixedHeaders();
  AddHandler(new CHandlerConn);
  AddHandler(new CHandlerLocation);
  AddHandler(new CHandlerKeepAlive);
  AddHandler(new CHandlerProxyConn);

  OnCreate();
  return OI_OK;
}

/*recycle the request*/
void
CDavRequest::Reset()
{
  if(m_pSession) {
    /* don't want to be knocked out while
     * we're doing something
     */
    if(m_pSession->m_pActiveRequest == this)
      return;
  }
  m_enuType = T_REQ_INVALID;
  m_strMethod.erase();
  m_mapReqHeaders.clear();
  m_mapRespHeaders.clear();
  ClearHandlers();

  m_pSession = NULL;
  m_pRBProvider = NULL;
  m_pPBConsumer = NULL;

  m_HttpStatus.Clear();
}

OI_RESULT 
CDavRequest::build()
{
  /*TODO: error-check for host settings*/
  OI_STRSTREAM_A ssHeaderBuf;
  CDavHost& Server = m_pSession->m_Server;
  OI_STRING_A strUTF8URI = m_strURI.UTF8();

  ssHeaderBuf << m_strMethod << " ";

  if(m_pSession->IsUsingProxy() &&
     //m_pSession->IsUsingSSL() &&
     strUTF8URI[0] == '/'){
    if (m_pSession->IsUsingSSL()) 
	{
		ssHeaderBuf << "https://";
	}
	else
	{
		ssHeaderBuf << "http://";
	}

    ssHeaderBuf << Server.GetHostName() << ":"
                << Server.GetPort();
  }

  ssHeaderBuf << Escape(strUTF8URI)
	      << " HTTP/1.1\r\n"
              << "Host: " << Server.GetHostName()
              << ":" << Server.GetPort() << "\r\n";

  if(m_bExpect100)
    ssHeaderBuf << "Expect: 100-continue\r\n";

  OI_HEADERMAP::iterator it;
  for(it = m_mapReqHeaders.begin();
      it != m_mapReqHeaders.end();
      it++){
    int nValCount = it->second.size();
    int nValIndex;
    for(nValIndex = 0; nValIndex < nValCount; nValIndex++) {
      ssHeaderBuf << it->first << ": " << it->second[nValIndex] << "\r\n";
    }
  }

  ssHeaderBuf << "\r\n";
  m_strHeaderBuf = ssHeaderBuf.str().c_str();
  return OI_OK;
}

OI_RESULT 
CDavRequest::discard_headers()
{
  OI_RESULT enuRet;
  size_t unLen = OI_GENBUFSIZE;

  CDavSocket* pSocket = m_pSession->getSocket();
  do{
    enuRet = pSocket->ReadLine(m_szBuffer, &unLen);
    if(OI_OK != enuRet) return enuRet;
    
    unLen = OI_GENBUFSIZE;
  } while(strcmp(m_szBuffer, "\r\n") != 0);

  return OI_OK;
}

OI_RESULT 
CDavRequest::discard_body(CDavResponseBody* pBody)
{
  OI_RESULT enuRet;
  size_t unReadLen, unBufLen = OI_GENBUFSIZE;
  for(enuRet = OI_RETRY; enuRet == OI_RETRY;){
    enuRet = pBody->Read(m_szBuffer, unBufLen, &unReadLen);
  }
  return enuRet;
}

OI_RESULT 
CDavRequest::read_header_line(char* pszBuf, const size_t unLen)
{
  OI_RESULT enuRet;
  size_t unBufLen = unLen;	//save buffer length
  size_t n = unBufLen;
  CDavSocket* pSocket = m_pSession->getSocket();
	
  enuRet = pSocket->ReadLine(pszBuf, &n);
  if(OI_OK != enuRet )
    return enuRet;
	
  stripEOL(pszBuf, &n);

  if(n==0){
    // No continuation of this header: stop reading
    return OI_OK;
  }

  pszBuf += n;
  unBufLen -= n;

  while(unBufLen > 0){
    size_t unReadLen = 1;
    char ch;
    enuRet = pSocket->Peek(&ch, &unReadLen);
    if(OI_OK != enuRet){
      //request will be aborted
      disconnect();
      return enuRet;
    }
    if (ch != ' ' && ch != '\t'){
      // No continuation of this header: stop reading. 
      return OI_RETRY;
    }
		
    n = unBufLen; 
    // Otherwise, read the next line onto the end of 'buf'.
    enuRet = pSocket->ReadLine(pszBuf, &n);
    if(OI_OK != enuRet)
      return enuRet;

    stripEOL(pszBuf, &n);
		
    // assert(buf[0] == ch), which implies len(buf) > 0.
    // Otherwise the TCP stack is lying, but we'll be paranoid.
    // This might be a \t, so replace it with a space to be
    // friendly to applications (2616 says we MAY do this). 
    if (n){
      pszBuf[0] = ' ';
    }

    // ready for the next header. 
    pszBuf += n;
    unBufLen -= n;
  }

  //response header line too long
  return OIHELINETOOLONG;
}

/**
 * Internal function. Read http headers line by line and store them in
 * map. When finished, call process_read_headers() to process
 * them. Notice that header is not required to be unique, so the same
 * header may appear several times in a response. And according to
 * rfc2068, headers may even appear a second time, while the body is
 * in chunked mode, in an optional footer behind last chunk.  Each
 * header identified by this function will have at least one lower
 * case name and one whitespace-stripped value in m_mapRespHeaders.
 */
OI_RESULT 
CDavRequest::read_resp_headers()
{
  OI_DEBUG("------------------------------\n");
  OI_RESULT enuRet;
  OI_STRING_A strName, strValue;

  int nColon, nCount = 0;
  char szHeader[OI_MAX_HDR_LEN];
  size_t unHeaderLen = OI_MAX_HDR_LEN;


  for( ;(enuRet = read_header_line(szHeader, unHeaderLen)) == OI_RETRY
	 && ++nCount < OI_MAX_HDR_FLD ;){
    strName = szHeader;
    TrimLeftA(strName);
    TrimRightA(strName);

    nColon = (int)strName.find_first_of(':');

    if(nColon < 0)
      continue;

    strValue = strName.substr(nColon + 1);
    strName.erase(nColon);

    TrimRightA(strName);
    TrimLeftA(strValue);

    OI_DEBUG("%20s\t%s\n", strName.c_str(), strValue.c_str());

    MakeLowerA(strName);
    /*store value in map*/
    m_mapRespHeaders[strName].push_back(strValue);
  }
  OI_DEBUG("\n");

  if(nCount >= OI_MAX_HDR_FLD)
    return OIHETOOMANYHEADERS;

  /*process headers*/
  enuRet = process_resp_headers();

  return enuRet;
}

OI_RESULT
CDavRequest::process_resp_headers()
{
  OI_RESULT enuRet = OI_OK;
  OI_HEADERMAP::iterator itHdr;
  OI_STRING_A strName;
  for(itHdr = m_mapRespHeaders.begin();
      itHdr != m_mapRespHeaders.end();
      itHdr++) {
    strName = itHdr->first;
    /*let's see if we have a handler for it*/
    CResponseHeaderHandler* pHandler;
    if(strName.length() > 0){
      OI_HANDLERMAP::iterator it = m_mapRespHeaderHandlers.find(strName);
      if(it != m_mapRespHeaderHandlers.end()){
	pHandler = it->second;
	enuRet = pHandler->Execute(this, itHdr->second);
      }	else {
	enuRet = m_pSession->ProcessExtraHeader(this,
						strName.c_str(),
						itHdr->second);
      }
      /*if handler says no, we abort*/
      if(OI_OK != enuRet)
	break;
    }
  }
  return enuRet;
}

OI_RESULT 
CDavRequest::read_status_line(bool bRetry)
{
  OI_RESULT enuRet;
  CDavSocket* pSocket = m_pSession->getSocket();
  size_t unLen = OI_GENBUFSIZE;

  enuRet = pSocket->ReadLine(m_szBuffer, &unLen);

  if(OI_OK != enuRet) 
    return RETRY_RET(bRetry, enuRet, enuRet);

  stripEOL(m_szBuffer, &unLen);

  enuRet = m_HttpStatus.Parse(m_szBuffer);
  OI_DEBUG("\tStatus: %d %s\n", m_HttpStatus.m_nCode, 
	   m_HttpStatus.m_strReason.c_str());
	
  if(OI_OK != enuRet)
    return enuRet;

  return OI_OK;
}

OI_RESULT 
CDavRequest::send()
{
  OI_RESULT enuRet;
  //OI_STRSTREAM_A ssCLen;
  OI_SIZE_T unBodyLen;
  bool bRetry;

  m_strHeaderBuf.erase();

  enuRet = m_pRBProvider->Open();
  if(OI_OK == enuRet){
    m_pRBProvider->Rewind();
    unBodyLen = m_pRBProvider->GetBodyLength();
    
    //ssCLen << unBodyLen;
    //SetRequestHeader("Content-Length", ssCLen.str().c_str());
    char chCLen[30]; //should be long enough for 64 bit
    sprintf(chCLen, OI_SIZEFORMAT, unBodyLen);
    SetRequestHeader("Content-Length", chCLen);
  } else {
    return enuRet;
  }

  /*pre build handler call*/
  OnPreBuild();
  enuRet = build();

  if(OI_OK != enuRet)
    return enuRet;

  enuRet = connect();
  if(OI_OK != enuRet)
    return enuRet;

  bRetry = m_pSession->IsPersistent();
  CDavSocket* pSocket = m_pSession->getSocket();
  pSocket->ResetBuffer();

  size_t unWritten = (size_t)m_strHeaderBuf.length();

  enuRet = pSocket->Write(m_strHeaderBuf.c_str(), &unWritten);

  if(OI_OK != enuRet){
    m_pRBProvider->Close();
    return RETRY_RET(bRetry, enuRet, enuRet);
  }

  /* send body in normal mode */
  if(unBodyLen > 0 && !m_bExpect100){
    enuRet = send_body(m_pRBProvider);
    if(OI_OK != enuRet){
      m_pRBProvider->Close();
      return RETRY_RET(bRetry, enuRet, enuRet);
    }
  }

  /* read status line */
  while((enuRet = read_status_line(bRetry)) == OI_OK 
	&& m_HttpStatus.m_nCodeClass == 1){
    /* send body in 100-continue mode */
    bRetry = false;
    if((enuRet = discard_headers()) != OI_OK)
      break;
    if(m_bExpect100 && m_HttpStatus.m_nCode == 100){
      enuRet = send_body(m_pRBProvider);
      if(OI_OK != enuRet)
	break;
    }
  }

  m_pRBProvider->Close();

  return enuRet;
}

OI_RESULT 
CDavRequest::send_body(CRequestBodyProvider* pRBProvider)
{
  OI_RESULT enuRet;

  CDavSocket* pSocket = m_pSession->getSocket();
  OI_SIZE_T unProgress = 0;
  OI_SIZE_T unTotal = pRBProvider->GetBodyLength(); 
  size_t unReadLen = OI_GENBUFSIZE;

  pRBProvider->Rewind();

  while((enuRet = pRBProvider->Read(this, m_szBuffer, &unReadLen)) 
	== OI_RETRY){
    unProgress += unReadLen;
    if(! m_pSession->OnSendProgress(this, unProgress, unTotal)){
      pRBProvider->OnCancel(this);
      m_pSession->OnCancelSend(this, pRBProvider);
      m_pSession->Disconnect();
      return OI_USERCANCELED;
    }

    enuRet = pSocket->Write(m_szBuffer, &unReadLen);
    if(OI_OK != enuRet)
      return enuRet;

    //restore the buffer size variable
    unReadLen = OI_GENBUFSIZE;
  }
  
  return enuRet;
}


OI_RESULT 
CDavRequest::begin(CDavResponseBody* pBody)
{
  OI_RESULT enuRet;
  int nRetry;
	
  m_bExpect100 = (m_pSession->IsExpecting100()
		  && m_pRBProvider->GetBodyLength() > OI_EXP_100_LEN
		  && m_pSession->IsHttp11());

  m_mapRespHeaders.clear();

  OnPreSend();
  enuRet = send();

  for(nRetry = 0;
      nRetry < OI_SOCK_MAXRETRY && enuRet == OI_RETRY
	&& m_pSession->IsPersistent();
      nRetry++){
    //DEBUG("Persistent connection timeout, retrying...\n")
    disconnect();
    enuRet = send();
  }

  OnPostSend();

  if((OI_RETRY == enuRet) && (nRetry >= OI_SOCK_MAXRETRY))
    return OISECLOSED;

  if(OI_OK != enuRet)
    return enuRet;

  //send() called read_status_line() so the return code is now available
  //check if session is HTTP 1.1 and thus can be set to persistent
  if(m_HttpStatus.m_nMajorVersion > 1 ||
     (m_HttpStatus.m_nMajorVersion == 1 && m_HttpStatus.m_nMinorVersion > 0)){
    m_pSession->SetHttp11(true);
  } else {
    m_pSession->SetHttp11(false);
    SetPersistent(false);
  }

  m_pSession->m_nLastStatusCode = m_HttpStatus.m_nCode;
  enuRet = read_resp_headers();
  if(OI_OK != enuRet)
    return enuRet;

  if(m_enuType == T_REQ_CONNECT && m_HttpStatus.m_nCodeClass == 2){
    pBody->m_enuMode = M_RSP_NO_BODY;
    SetPersistent(true);
  }
	
  if(m_enuType == T_REQ_HEAD || 
     m_HttpStatus.m_nCode == 204 || 
     m_HttpStatus.m_nCode == 304){
    pBody->m_enuMode = M_RSP_NO_BODY;
  }

  return OI_OK;
}

OI_RESULT 
CDavRequest::end(CDavResponseBody* pBody)
{
  OI_RESULT enuRet = OI_OK;

  if(S_CONN_CONNECTED == m_pSession->GetConnectState())
    if(M_RSP_CHUNKED == pBody->m_enuMode)
      enuRet = read_resp_headers();

  return enuRet;
}

OI_RESULT 
CDavRequest::connect()
{
  return m_pSession->Connect(this);
}

void 
CDavRequest::disconnect()
{
  m_pSession->Disconnect();
}

void 
CDavRequest::softDisconnect()
{
  if(!m_pSession->IsPersistent())
    disconnect();
}

OI_RESULT 
CDavRequest::Dispatch(CRequestBodyProvider* pRBProvider,
		      CResponseBodyConsumer* pPBConsumer)
{
  OI_RESULT enuRet;

  /* is this request created properly?*/
  if(!m_pSession)
    return OIGEINVALIDSTATE;

  /*only one request can be processed at a time*/
  if(m_pSession->m_pActiveRequest)
    return OIGEINVALIDSTATE;

  m_tmLastDispatch = time(NULL);

  m_pSession->m_pActiveRequest = this;
  m_pSession->m_nLastStatusCode = -1;
  m_pSession->m_strErrorDescription.erase();

  for(enuRet = OI_RETRY; enuRet == OI_RETRY;){
    /*if this is true, body is not read, we should disconnect*/
    bool bSimpleError = false;

    CDavResponseBody body(this, m_pSession->getSocket());
    CDavXmlBuffer* pXmlBuffer = m_pSession->GetXmlBuffer();
    CDavMemBuffer* pMemBuffer = m_pSession->GetMemBuffer();
    m_pRBProvider = pRBProvider;
    m_pPBConsumer = pPBConsumer;

    pXmlBuffer->Clear();
    pMemBuffer->Clear();

    enuRet = begin(&body);
    if(enuRet != OI_OK)
      break;
    /* headers are processed after begin() */

    /* check content type
     * notice Content-Type header is not always required here because
     * this is a generic function and we have server policy mechanism.
     * we don't check Content-Type every time.
     */
    OI_STRING_A strCType;
    OI_STRING_A strHeaderName = OI_RSPHDR_CTYPE;	//Content-Type
    MakeLowerA(strHeaderName);
    OI_HEADERMAP::iterator itHeader = m_mapRespHeaders.find(strHeaderName);
    /* do we have a Content-Type header? */
    if(itHeader != m_mapRespHeaders.end()) {
      /* this is guarenteed by read_resp_header() */
      OI_ASSERT(itHeader->second.size() > 0);
      strCType = itHeader->second[0];
    }

    body.m_strContentType = strCType;

    /* get server string from headers*/
    m_pSession->m_strServerString.erase();
    strHeaderName = OI_RSPHDR_SERVER;
    MakeLowerA(strHeaderName);
    itHeader = m_mapRespHeaders.find(strHeaderName);
    if(itHeader != m_mapRespHeaders.end()) {
      /* this is guarenteed by read_resp_header() */
      OI_ASSERT(itHeader->second.size() > 0);
      m_pSession->m_strServerString = itHeader->second[0];
    }

    //chop off tail after semicolon
    OI_STRING_A::size_type nSemiColon = strCType.find(';');
    if(nSemiColon != OI_STRING_A::npos)
      strCType.erase(nSemiColon);

    int nStatus = m_HttpStatus.m_nCode;
    CDavServerPolicy* pPolicy = m_pSession->m_pServerPolicy;
    OIOPERATION opr;

    enuRet = pPolicy->CheckResponse(m_pSession,
				    m_pSession->m_strServerString.c_str(),
				    m_strMethod.c_str(), 
				    nStatus, strCType.c_str(),
				    &opr);
    if(enuRet != OI_OK)
      break;

    m_pSession->m_enuLastOperation = opr.enuOpCode;
    enuRet = opr.enuErrCode;
    OI_RESULT enuPullResult = OI_OK;
    switch(opr.enuOpCode){
    case OP_CACHE_HTML:
      {
	CPBCSMMemBuffer bufWriter(this);
	//?call OnPreRecv?
	enuPullResult = bufWriter.PullResponseBody(&body);
	//?call OnPostRecv?
	if(OI_OK != enuPullResult)
	  enuRet = enuPullResult;
	break;
      }
    case OP_PARSE_ERR_XML:
      {
	if(opr.szCutElement[0] == '\0')
	  strcpy(opr.szCutElement, OI_XML_CUTELEMENT); //"response"
	CPBCSMXml errorConsumer(this, opr.szCutElement, pXmlBuffer);
	enuPullResult = errorConsumer.PullResponseBody(&body);
	if(OI_OK != enuPullResult)
	  enuRet = enuPullResult;
	break;
      }
    case OP_SIMPLE_ERR:
      {
	enuRet = opr.enuErrCode;
	bSimpleError = true;
	break;
      }
    case OP_USE_DEFAULT:
      {
	if(M_RSP_NO_BODY == body.m_enuMode){
	  enuPullResult = OI_OK;
	  break;
	}
	enuPullResult = m_pPBConsumer->CheckCType(strCType.c_str());
	if(OI_OK != enuPullResult){
	  /*type mismatch, discard body and return the error code*/
	  enuRet = enuPullResult;
	  enuPullResult = discard_body(&body);
	} else {
	  enuPullResult = m_pPBConsumer->Open();
	  if(OI_OK != enuPullResult){
	    enuRet = enuPullResult;
	    break;
	  }
	    
	  OnPreRecv();
	  enuPullResult = m_pPBConsumer->PullResponseBody(&body);
	  OnPostRecv();
	  
	  m_pPBConsumer->Close();
	}
	break;
      }
    case OP_AUTHENTICATE:
      {
	/* for the HEAD mess */
	if(M_RSP_NO_BODY == body.m_enuMode){
	  enuPullResult = OI_OK;
	} else {
	  CPBCSMMemBuffer bufWriter(this);
	  //call OnPreRecv() ?
	  enuPullResult = bufWriter.PullResponseBody(&body);
	  //call OnPostRecv() ?
	  if(OI_OK != enuPullResult){
	    enuRet = enuPullResult;
	    break;
	  }
	}
	
	enuRet = m_pSession->m_pAuthManager->QueryEndRequest(this);
	break;
      }
    default:
      /*this case should first be avoided by developper in debug build*/
      OI_ASSERT(false);
      bSimpleError = true;
      /* this is a fatal error, abort */
      enuPullResult = enuRet = OIYEILLEGALOPCODE;
      break;
    }
    

    if(bSimpleError) {
      disconnect();
      break;
    } else if(OI_OK != enuPullResult){
      //something's wrong, disconnect immediately
      disconnect();
      enuRet = enuPullResult;
      break;
    } else {
      //do the cleanup
      OI_RESULT enuEndResult = end(&body);
      if(OI_OK != enuEndResult){
	enuRet = enuEndResult;
	break;
      }
      softDisconnect();
    }
  }
  softDisconnect();
  m_pSession->m_pActiveRequest = NULL;
  OI_DEBUG("elapsed time: %d second(s)\n", time(NULL) - m_tmLastDispatch);
  return enuRet;
}

void 
CDavRequest::OnCreate()
{
  m_pSession->m_pAuthManager->OnCreateRequest(this);
  m_pSession->OnCreateRequest(this);
}

void 
CDavRequest::OnDestroy()
{
  m_pSession->OnDestroyRequest(this);
  m_pSession->getSocket()->ResetBuffer();
}

void
CDavRequest::OnPreBuild()
{
  m_pSession->OnPreBuildRequest(this);
}

void
CDavRequest::OnPreSend()
{
  m_pSession->m_pAuthManager->OnPreSendRequest(this);
  m_pSession->OnPreSendRequest(this);
}

void 
CDavRequest::OnPostSend()
{
  m_pSession->OnPostSendRequest(this);
}

void 
CDavRequest::OnPreRecv()
{
  m_pSession->m_unResponseLength = 0;
  m_pSession->OnPreRecvResponse(this);
  m_pSession->m_nMaxConnections--;
}

void 
CDavRequest::OnPostRecv()
{
  m_pSession->OnPostRecvResponse(this);
}

#undef RETRY_RET

