/* 
   Http request handler
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 1999-2002, Joe Orton <joe@manyfish.co.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavRequest.h 132 2005-06-24 09:09:43Z komat $
*/

#ifndef ONION_HEADER_DAVREQUEST_H
#define ONION_HEADER_DAVREQUEST_H 1

#include <time.h>
#include <onion/OnionCfg.h>
#include <onion/DavHttpStatus.h>
#include <map>
#include <vector>

//forward declarations
class CDavSocket;
class CDavResponseBody;
class CResponseHeaderHandler;
class CDavWorkSession;
class CRequestBodyProvider;
class CResponseBodyConsumer;

typedef std::vector<OI_STRING_A> OI_STRLIST_A;
/**map of request or response headers
  */
typedef std::map<OI_STRING_A, OI_STRLIST_A> OI_HEADERMAP;
typedef std::map<OI_STRING_A, CResponseHeaderHandler*> OI_HANDLERMAP;

/** CDavRequest is the core of HTTP processing.
 *
 * CDavRequest handles all operations during the lifespan of a single
 * HTTP request. Under most circumstances, you don't create request
 * object explicitly. Member functions of CDavWorkSession will do it
 * for you. Simple applications don't even have to play with this
 * class. \n
 *
 * However, programs that need to have more feed back and custom
 * control over DAV/HTTP operations may find this class useful.
 * Pointer to request object appears in many of the handler functions
 * of CDavWorkSession. By overriding these functions, programs can
 * access and/or manipulate a request at certain phases of the whole
 * process.
 * 
 */
OI_BEGIN_CLASSEXPORT

class OI_EXPORT CDavRequest
{
 public:

  /** @name Constructor */

  /* @{ */
  /**Constructor
   */
  CDavRequest();
  /* @} */

  /** @name Destructor */

  /* @{ */
  /**Destructor
   */
  virtual ~CDavRequest();
  /* @} */

  /**add a response header handler
   *
   * Inserts pointer of the handler to request's handler map
   * #m_mapRespHeaderHandlers. If a handler with the same name exists,
   * old one will be deleted. The handler must be created on heap
   * rather than on stack. Because all header handlers will be deleted
   * when request object is destroyed.
   *
   * @param pHandler pointer(not NULL) to the handler to be added,
   *	must be created on heap.
   * @param pszName pointer to header name string. if NULL, handler's
   *	GetHeaderName() will be consulted for the name.
   *
   * @return 'true' if a handler with same name was found, 'false' otherwise.
   *
   * @see #m_mapRespHeaderHandlers
   */
  bool AddHandler(CResponseHeaderHandler* pHandler, const char *pszName = 0);

  /**delete response header handler
   *
   * Deletes header handler with the name specified with \e pszName
   *
   * @param pszName pointer(not NULL) to header name string
   *
   * @return 'true' if a handler with same name was found, 'false'
   *	otherwise.
   *
   * @see #m_mapRespHeaderHandlers
   */
  bool RemoveHandler(const char* pszName);

  /**clear handler map
   *
   * Clears handler map #m_mapRespHeaderHandlers and delete all
   * handlers in it
   */
  void ClearHandlers();

  /**add a request header
   *
   * Inserts a header into request header map #m_mapReqHeaders. Most
   * of the time one header only have one corresponding value. But
   * multiple value is OK, too. So this function won't erase existing
   * values. To overwrite any existing value, use #SetRequestHeader.
   *
   * @param pszName pointer(not NULL) to header name string
   * @param pszValue pointer to header value string. If NULL, empty
   *	will be used.
   *
   * @see #SetRequestHeader
   */
  void AddRequestHeader(const char* pszName, const char* pszValue);

  /**set a request header
   *
   * Similiar to #AddRequestHeader but existing header value will be
   * erased.
   *
   * @param pszName pointer(not NULL) to header name string
   * @param pszValue pointer to header value string. If NULL, empty
   *	will be used.
   *
   * @see #AddRequestHeader
  */
  void SetRequestHeader(const char* pszName, const char* pszValue);

  /**remove a request header
   *
   * Removes a request header if it exists.
   * 
   * @param pszName(not NULL) pointer to header name string
   *
   * @return 'true' if the header exists, 'false' otherwise
   */
  bool RemoveRequestHeader(const char* pszName);

  /**clear request headers
   *
   * Removes all request headers.
   */
  void ClearRequestHeaders();

  /**set persistent
   *
   * Set session as persistent or not
   *
   * @param bValue whether to set session as persistent
   */
  void SetPersistent(bool bValue);

  /**get method type
   *
   * Gets type of this request
   *
   * @return method type in enumeration
   */
  OI_REQ_TYPE GetMethod();

  /**get method string
   *
   * Gets the string of this method. This is what exactly sent to the
   * server.
   *
   * @return pointer to method string
   */
  const char* GetMethodStr();

  /**create request
   * 
   * Before a request can be dispatched, it must be first created.
   * 
   * @param pSession pointer(not NULL) to the session in which this
   *	request will be created.
   * @param enuType type of this request. Any value other than 
   *	\e T_REQ_CONNET shall be OK.
   * @param pszMethod pointer(not NULL) to method name string.
   *	'CONNECT' is not recomended.
   * @param strURI URI of the resource
   *
   * @return an error code
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   * @see #Dispatch
   * @see #Reset
   */
  OI_RESULT Create(CDavWorkSession* pSession,
		   OI_REQ_TYPE enuType,
		   const char *pszMethod,
		   const X& strURI);

  /**reset a request before or after dispatching
   *
   * This function does not have to be called directly
   */
  void Reset();

  /**dispatch a request
   *
   * This function actually sends out request and receives response.
   *
   * @param pRBProvider pointer(not NULL) to a request body provider
   * @param pPBConsumer pointer(not NULL) to a response body consumer
   *
   * @return an error code
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   * @see #Create
   * @see #Reset
   */
  OI_RESULT Dispatch(CRequestBodyProvider* pRBProvider,
		     CResponseBodyConsumer* pPBConsumer);
  CDavWorkSession* GetSession();
  int GetStatusClass();
  int GetStatusCode();
 protected:
  virtual void OnCreate();
  virtual void OnDestroy();
  virtual void OnPreBuild();
  virtual void OnPreSend();
  virtual void OnPreRecv();
  virtual void OnPostSend();
  virtual void OnPostRecv();

  OI_RESULT begin(CDavResponseBody* pBody);
  OI_RESULT end(CDavResponseBody* pBody);
  void softDisconnect();
 private:
  void addFixedHeaders();
  void stripEOL(char* pszBuf, size_t* punLen);
  OI_RESULT connect();
  void disconnect();
  OI_RESULT build();
  OI_RESULT discard_headers();
  OI_RESULT discard_body(CDavResponseBody* pBody);
  OI_RESULT read_header_line(char* pszBuf, size_t unLen);
  OI_RESULT read_resp_headers();
  OI_RESULT process_resp_headers();
  OI_RESULT read_status_line(bool bRetry);
  OI_RESULT send();
  OI_RESULT send_body(CRequestBodyProvider* pRBProvider);
 public:
  OI_HEADERMAP m_mapReqHeaders;		//request headers
  OI_HEADERMAP m_mapRespHeaders;	//response headers
  OI_HANDLERMAP m_mapRespHeaderHandlers;//handlers for response headers
  char m_szBuffer[OI_GENBUFSIZE];	//request local buffer
  X m_strURI;
 protected:
  OI_STRING_A m_strMethod;
  OI_REQ_TYPE m_enuType;
  OI_STRING_A m_strHeaderBuf;		//request header string buffer
  CDavHttpStatus m_HttpStatus;
  bool m_bExpect100;			//local cache of session variable
  CRequestBodyProvider* m_pRBProvider;
  CResponseBodyConsumer* m_pPBConsumer;
  CDavWorkSession* m_pSession;
  time_t m_tmLastDispatch;		//for debug output
};

OI_END_CLASSEXPORT

#endif /*ONION_HEADER_DAVREQUEST_H*/
