/* 
   Dav resource infomation container
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavResourceNode.cpp 557 2017-10-29 01:09:14Z tfbuilder $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/DavResourceNode.h>
#include <stdio.h>

CDavResourceNode::CDavResourceNode()
{
  Reset();
}

CDavResourceNode::~CDavResourceNode()
{
}

void
CDavResourceNode::Reset()
{
  m_ulValidProps = 0L;
  m_ulPropsToParse = OI_RNP_CONVENTIONAL;
  m_enuResourceType = T_RES_NORMAL;
  m_unContentLength = 0;
  m_tmCreationDate = -1;
  m_tmLastModified = -1;
  m_unQuota = 0;
  m_unQuotaUsed = 0;
  m_strContentLanguage.erase();
  m_strContentType.erase();
  m_strDisplayName.erase();
  m_strETag.erase();
  m_strURI.erase();
  m_mapPropStats.clear();
  m_vecLocklist.clear();
}

X&
CDavResourceNode::URI()
{
  return m_strURI;
}

OI_RES_TYPE
CDavResourceNode::GetResourceType()
{
  return m_enuResourceType;
}

X&
CDavResourceNode::DisplayName()
{
  return m_strDisplayName;
}

OI_SIZE_T
CDavResourceNode::GetContentLength()
{
  return m_unContentLength;
}

time_t
CDavResourceNode::GetLastModified()
{
  return m_tmLastModified;
}

time_t
CDavResourceNode::GetCreationDate()
{
  return m_tmCreationDate;
}

const char*
CDavResourceNode::GetETag()
{
  return m_strETag.c_str();
}

const char*
CDavResourceNode::GetContentType()
{
  return m_strContentType.c_str();
}

const char*
CDavResourceNode::GetContentLanguage()
{
  return m_strContentLanguage.c_str();
}

OI_SIZE_T
CDavResourceNode::GetQuota()
{
  return m_unQuota;
}

OI_SIZE_T
CDavResourceNode::GetQuotaUsed()
{
  return m_unQuotaUsed;
}

unsigned long
CDavResourceNode::GetFlags()
{
  return m_ulValidProps;
}

OI_RESULT
CDavResourceNode::Parse(XNS(DOMNode)* pResponse, unsigned long ulPropsToParse)
{
  OI_ASSERT(pResponse);

  m_ulPropsToParse = ulPropsToParse;

  if(XNS(DOMNode)::ELEMENT_NODE != pResponse->getNodeType())
    return OIRESYNTAX;	//TODO: better error codes

  XNS(DOMNode) *pEnum, *pProp;
  XNS(DOMNodeList) *pNodeList, *pPropNodeList;
  X strDavNS(OI_XML_DAVNAMESPACE);

  //getting href
  pNodeList =
    ((XNS(DOMElement)*)pResponse)->getElementsByTagNameNS(strDavNS, X("href"));
  if(pNodeList->getLength() < 1) 
    return OIREHREFNOTFOUND;

  pEnum = pNodeList->item(0);
  pEnum = pEnum->getFirstChild();
  if(!pEnum) 
    return OIREHREFNOTFOUND;
  if(XNS(DOMNode)::TEXT_NODE != pEnum->getNodeType()) 
    return OIREHREFNOTFOUND;

  m_strURI = (XMLByte*)(Unescape((const char*)X(pEnum->getNodeValue()))).c_str();
	

  //getting propstat list and parse it
  pNodeList =
    ((XNS(DOMElement)*)pResponse)->getElementsByTagNameNS(strDavNS, X("propstat"));
  int nLength = pNodeList->getLength();

  for(int nIndex = 0; nIndex < nLength; nIndex++){
    pEnum = pNodeList->item(nIndex);
    pPropNodeList =
      ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(strDavNS, X("prop"));
    pProp = pPropNodeList->item(0);

    if(!pProp)
      continue;

    int nStatus = getStatusCode(pEnum);

    if(nStatus == 200){
      parseProp(pProp);
      removeUnprocessed(pProp, 200);
    } else if(nStatus < 0) {
      continue;
    } else {
	removeUnprocessed(pProp, nStatus);
    }

    PROPSTATMAP::iterator it = m_mapPropStats.find(200);
    if(it != m_mapPropStats.end()){
      //do we have unparsed 200 properties?
      if(it->second->hasChildNodes())
	m_ulValidProps |= OI_RNP_EXTRAPROPERTIES;
    }
  }

  return OI_OK;
}

XNS(DOMNode)*
CDavResourceNode::GetProp(int nStatusCode)
{
  PROPSTATMAP::iterator it = m_mapPropStats.find(nStatusCode);
  if(it == m_mapPropStats.end())
    return NULL;
  else
    return it->second;
}

int
CDavResourceNode::GetLockCount()
{
  return m_vecLocklist.size();
}

CDavLock*
CDavResourceNode::GetLock(unsigned int unIndex)
{
  if(unIndex >= m_vecLocklist.size())
    return NULL;
  else
    return &m_vecLocklist[unIndex];
}

int
CDavResourceNode::getStatusCode(XNS(DOMNode)* pStatus)
{
  if(!pStatus) return -1;
  if(XNS(DOMNode)::ELEMENT_NODE != pStatus->getNodeType()) return -1;

  XNS(DOMNodeList)* pNodeList =
    ((XNS(DOMElement)*)pStatus)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
							X("status"));

  XNS(DOMNode)* pEnum = pNodeList->item(0);
  if(!pEnum)
    return -1;
  if(XNS(DOMNode)::ELEMENT_NODE != pEnum->getNodeType())
    return -1;

  pEnum = pEnum->getFirstChild();
  if(!pEnum)
    return -1;
  if(XNS(DOMNode)::TEXT_NODE != pEnum->getNodeType())
    return -1;
  OI_STRING_A strStatus = (const char*)X(pEnum->getNodeValue());
  TrimLeftA(strStatus);
  TrimRightA(strStatus);

  int nMajor, nMinor, nStatus;
  if(sscanf(strStatus.c_str(), 
	    "HTTP/%d.%d %3d",
	    &nMajor,
	    &nMinor,
	    &nStatus) != 3)
    return -1;
  return nStatus;
}

void
CDavResourceNode::parseProp(XNS(DOMNode)* pProp)
{
  if(m_ulPropsToParse & OI_RNP_RESOURCETYPE)
    parseResourceType(pProp);
  if(m_ulPropsToParse & OI_RNP_CONTENTLENGTH)
    parseContentLength(pProp);
  if(m_ulPropsToParse & OI_RNP_DISPLAYNAME)
    parseDisplayName(pProp);
  if(m_ulPropsToParse & OI_RNP_LASTMODIFIED)
    parseLastModified(pProp);
  if(m_ulPropsToParse & OI_RNP_CREATIONDATE)
    parseCreationDate(pProp);
  if(m_ulPropsToParse & OI_RNP_ETAG)
    parseETag(pProp);
  if(m_ulPropsToParse & OI_RNP_CONTENTTYPE)
    parseContentType(pProp);
  if(m_ulPropsToParse & OI_RNP_CONTENTLANGUAGE)
    parseContentLanguage(pProp);
  if(m_ulPropsToParse & OI_RNP_LOCKSUPPORTED)
    parseSupportedLock(pProp);
  if(m_ulPropsToParse & OI_RNP_LOCK)
    parseActiveLock(pProp);
  if(m_ulPropsToParse & OI_RNP_QUOTA)
    parseQuota(pProp);
  if(m_ulPropsToParse & OI_RNP_QUOTAUSED)
    parseQuotaUsed(pProp);
}

void
CDavResourceNode::parseResourceType(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("resourcetype"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  m_ulValidProps |= OI_RNP_RESOURCETYPE;
  XNS(DOMNode)* pResourceType = pEnum->getFirstChild();
  if(pResourceType){
    if(XNS(DOMNode)::ELEMENT_NODE == pResourceType->getNodeType()){
      const XMLCh* pszResourceType = pResourceType->getLocalName();
      if(XNS(XMLString)::compareString(pszResourceType, X("collection")) == 0){
	m_enuResourceType = T_RES_COLLECTION;
      }
    }
  } else {
    m_enuResourceType = T_RES_NORMAL;
  }

  pProp->removeChild(pEnum);
}

void
CDavResourceNode::parseContentLength(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("getcontentlength"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  XNS(DOMNode)* pContentLength = pEnum->getFirstChild();
  if(pContentLength){
    if(XNS(DOMNode)::TEXT_NODE == pContentLength->getNodeType()){
      const XMLCh* pszValue = pContentLength->getNodeValue();
      if(pszValue) {
	X strContentLength(pszValue);
	char* pchPos = NULL;
	m_unContentLength = OI_STRTOUL((const char*)strContentLength,
				       &pchPos, 10);
	m_ulValidProps |= OI_RNP_CONTENTLENGTH;
      }
    }
  }
  pProp->removeChild(pEnum);
}

void
CDavResourceNode::parseDisplayName(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("displayname"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  XNS(DOMNode)* pDisplayName = pEnum->getFirstChild();
  if(pDisplayName){
    if(XNS(DOMNode)::TEXT_NODE == pDisplayName->getNodeType()){
      m_strDisplayName = pDisplayName->getNodeValue();
      m_ulValidProps |= OI_RNP_DISPLAYNAME;
    }
  }
  pProp->removeChild(pEnum);
}

void
CDavResourceNode::parseLastModified(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("getlastmodified"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  XNS(DOMNode)* pLastModified = pEnum->getFirstChild();
  if(pLastModified){
    if(XNS(DOMNode)::TEXT_NODE == pLastModified->getNodeType()){
      //m_tmLastModified = GmtToLocal(ParseTime(X(pLastModified->getNodeValue())));
      m_tmLastModified = ParseTime(X(pLastModified->getNodeValue()));
      m_ulValidProps |= OI_RNP_LASTMODIFIED;
    }
  }
  pProp->removeChild(pEnum);
}

void
CDavResourceNode::parseCreationDate(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("creationdate"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  XNS(DOMNode)* pCreationDate = pEnum->getFirstChild();
  if(pCreationDate){
    if(XNS(DOMNode)::TEXT_NODE == pCreationDate->getNodeType()){
      //m_tmCreationDate = GmtToLocal(ParseTimeISO8601(X(pCreationDate->getNodeValue())));
      m_tmCreationDate = ParseTimeISO8601(X(pCreationDate->getNodeValue()));
      m_ulValidProps |= OI_RNP_CREATIONDATE;
    }
  }
  pProp->removeChild(pEnum);
}

void
CDavResourceNode::parseETag(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("getetag"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  XNS(DOMNode)* pETag = pEnum->getFirstChild();
  if(pETag){
    if(XNS(DOMNode)::TEXT_NODE == pETag->getNodeType()){
      m_strETag =X(pETag->getNodeValue());
      m_ulValidProps |= OI_RNP_ETAG;
    }
  }
  pProp->removeChild(pEnum);
}

void
CDavResourceNode::parseContentType(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("getcontenttype"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  XNS(DOMNode)* pContentType = pEnum->getFirstChild();
  if(pContentType){
    if(XNS(DOMNode)::TEXT_NODE == pContentType->getNodeType()){
      m_strContentType =X(pContentType->getNodeValue());
      m_ulValidProps |= OI_RNP_CONTENTTYPE;
    }
  }
  pProp->removeChild(pEnum);
}

void
CDavResourceNode::parseContentLanguage(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("getcontentlanguage"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  XNS(DOMNode)* pContentLanguage = pEnum->getFirstChild();
  if(pContentLanguage){
    if(XNS(DOMNode)::TEXT_NODE == pContentLanguage->getNodeType()){
      m_strContentLanguage =X(pContentLanguage->getNodeValue());
      m_ulValidProps |= OI_RNP_CONTENTLANGUAGE;
    }
  }
  pProp->removeChild(pEnum);
}

void
CDavResourceNode::parseSupportedLock(XNS(DOMNode)* pProp)
{
  X pszDAV(OI_XML_DAVNAMESPACE);
  XNS(DOMNode)* pSupportedLock;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(pszDAV,
						      X("supportedlock"));
  pSupportedLock = pList->item(0);
  if(!pSupportedLock) 
    return;

  pList =
    ((XNS(DOMElement)*)pSupportedLock)->getElementsByTagNameNS(pszDAV,
							       X("lockentry"));
  int nLength = pList->getLength();

  for(int nIndex = 0; nIndex < nLength; nIndex++){
    XNS(DOMNode)* pLockEntry = pList->item(nIndex);
    XNS(DOMNodeList)* pScopeList =
      ((XNS(DOMElement)*)pLockEntry)->getElementsByTagNameNS(pszDAV,
							     X("lockscope"));
    XNS(DOMNode)* pScope = pScopeList->item(0);
    if(pScope){
      XNS(DOMNodeList)* pValueList =
	((XNS(DOMElement)*)pScope)->getElementsByTagNameNS(pszDAV, X("*"));
      XNS(DOMNode)* pValue = pValueList->item(0);
      if(pValue){
	const XMLCh* pszScope = pValue->getLocalName();
	if(XNS(XMLString)::compareString(pszScope, X("exclusive")) == 0)
	  m_ulValidProps |= OI_RNP_ELOCKSUPPORTED;
	else if(XNS(XMLString)::compareString(pszScope, X("shared")) == 0)
	  m_ulValidProps |= OI_RNP_SLOCKSUPPORTED;
      }
    }
  }
  pProp->removeChild(pSupportedLock);
}

void
CDavResourceNode::parseActiveLock(XNS(DOMNode)* pProp)
{
  X pszDAV(OI_XML_DAVNAMESPACE);
  XNS(DOMNode) *pEnum, *pLockDiscovery;
  XNS(DOMNodeList)* pList = 
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(pszDAV, X("lockdiscovery"));
  pLockDiscovery = pList->item(0);

  if(!pLockDiscovery)
    return;
  pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(pszDAV, X("activelock"));
  int nLength = pList->getLength();

  for(int nIndex = 0; nIndex < nLength; nIndex++){
    pEnum = pList->item(nIndex);
    
    //getting <DAV:locktoken>
    XNS(DOMNodeList)* pLockPropList =
      ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(pszDAV, X("locktoken"));
    XNS(DOMNode)* pLockProp = pLockPropList->item(0);
    if(!pLockProp)
      continue;
    //the <DAV:href> element
    const XMLCh* pszToken;
	pLockProp = ((XNS(DOMElement)*)pLockProp)->getElementsByTagNameNS(pszDAV, X("href"))->item(0);
    if(!pLockProp)
      continue;
    if(XNS(DOMNode)::ELEMENT_NODE != pLockProp->getNodeType())
      continue;
    //the text value of <DAV:href>
    XNS(DOMNode)* pValue = pLockProp->getFirstChild();
    if(!pValue)
      continue;
    if(XNS(DOMNode)::TEXT_NODE != pValue->getNodeType())
      continue;
    pszToken = pValue->getNodeValue();

    //getting <DAV:lockscope>
    OI_LOCK_SCOPE enuScope;
    pLockPropList =
      ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(pszDAV, X("lockscope"));
    pLockProp = pLockPropList->item(0);
    if(!pLockProp)
      continue;
    pValue = pLockProp->getFirstChild();
    if(!pValue)
      continue;
    if(XNS(DOMNode)::ELEMENT_NODE != pValue->getNodeType())
      continue;
    const XMLCh* pszScope = pValue->getLocalName();
    if(XNS(XMLString)::compareString(pszScope, X("exclusive")) == 0)
      enuScope = SCP_LOCK_EXCLUSIVE;
    else if(XNS(XMLString)::compareString(pszScope, X("shared")) == 0)
      enuScope = SCP_LOCK_SHARED;
    else
      continue;

    //getting <DAV:depth>
    OI_LOCK_DEPTH enuDepth;
    pLockPropList =
      ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(pszDAV, X("depth"));
    pLockProp = pLockPropList->item(0);
    if(!pLockProp)
      continue;
    pValue = pLockProp->getFirstChild();
    if(!pValue)
      continue;
    if(XNS(DOMNode)::TEXT_NODE != pValue->getNodeType())
      continue;
    const XMLCh* pszDepth = pValue->getNodeValue();
    if(!pszDepth)
      continue;
    if(XNS(XMLString)::compareString(pszDepth, X("0")) == 0)
      enuDepth = D_LOCK_ZERO;
    else if(XNS(XMLString)::compareString(pszDepth, X("infinity")) == 0)
      //TODO: In RFC it is Infinity, investigation needed
      enuDepth = D_LOCK_INFINITE;
    else
      continue;
 

    //getting <DAV:timeout>
    long lTimeout;
    pLockPropList =
      ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(pszDAV, X("timeout"));
    pLockProp = pLockPropList->item(0);
    if(!pLockProp)
      continue;
    pValue = pLockProp->getFirstChild();
    if(!pValue)
      continue;
    if(XNS(DOMNode)::TEXT_NODE != pValue->getNodeType())
      continue;
    const XMLCh* pszTimeout = pValue->getNodeValue();
    if(!pszTimeout)
      continue;
    OI_STRING_A strTimeout = (const char*)X(pszTimeout);
    if(1 != sscanf(strTimeout.c_str(), "Second-%ld", &lTimeout))
      //TODO: put the format into Def file
      lTimeout = 0;

    //TODO: is owner node required for a valid lock object?
    //getting <DAV:owner>
    XNS(DOMNode)* pOwner;
    pLockPropList =
      ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(pszDAV, X("owner"));
    pOwner = pLockPropList->item(0);
    
    //setting the flag
    unsigned int unMask = OI_LIVF_ALL;
    if(!pOwner)
      unMask &= ~OI_LIVF_OWNER;
    
    
    //we have everything, construct a lock object
    //and insert it into the list
    CDavLock theLock;
    theLock.m_strToken = X(pszToken);
    theLock.m_enuType = T_LOCK_WRITE;
    theLock.m_enuScope = enuScope;
    theLock.m_enuDepth = enuDepth;
    theLock.m_lTimeout = lTimeout;
    theLock.m_pOwner = pOwner;
    theLock.m_unMask = unMask;

    m_vecLocklist.push_back(theLock);
    m_ulValidProps |= OI_RNP_LOCK;
  }

  pProp->removeChild(pLockDiscovery);
}

bool
CDavResourceNode::parseLockStatic(XNS(DOMNode)* pNode, CDavLock* pLock)
{
  OI_ASSERT(pNode && pLock);

  pLock->m_unMask = 0;
  XNS(DOMNode)* pEnum = pNode;

  //processing element <DAV:activelock>
  XNS(DOMNodeList)* pLockPropList;
  XNS(DOMNode)* pLockProp, *pValNode;

  //getting the <DAV:locktoken> element
  pLockPropList =
    ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("locktoken"));
  pLockProp = pLockPropList->item(0);
  if(pLockProp){
    //getting the <DAV:href> element
	  pLockProp = ((XNS(DOMElement)*)pLockProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
							X("href"))->item(0);
    if(pLockProp){
      if(XNS(DOMNode)::ELEMENT_NODE == pLockProp->getNodeType()){
	XNS(DOMNode)* pTextNode = pLockProp->getFirstChild();
	if(pTextNode){
	  if(XNS(DOMNode)::TEXT_NODE == pTextNode->getNodeType()){
	    pLock->m_strToken = X(pTextNode->getNodeValue());
	    pLock->m_unMask |= OI_LIVF_TOKEN;
	  }
	}
      }
    }
  }

  //getting the <DAV:lockscope> element
  pLockPropList =
    ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("lockscope"));
  pLockProp = pLockPropList->item(0);
  if(pLockProp){
    pValNode = pLockProp->getFirstChild();
    if(pValNode){
      if(XNS(DOMNode)::ELEMENT_NODE == pValNode->getNodeType()){
	const XMLCh* pszScope = X(pValNode->getLocalName());
	if(XNS(XMLString)::compareString(pszScope, X("exclusive")) == 0){
	  pLock->m_enuScope = SCP_LOCK_EXCLUSIVE;
	  pLock->m_unMask |= OI_LIVF_SCOPE;
	} else if(XNS(XMLString)::compareString(pszScope, X("shared")) == 0) {
	  pLock->m_enuScope = SCP_LOCK_SHARED;
	  pLock->m_unMask |= OI_LIVF_SCOPE;
	}
      }
    }
  }

  //getting the <DAV:locktype> element
  pLockPropList =
    ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("locktype"));
  pLockProp = pLockPropList->item(0);
  if(pLockProp){
    pValNode = pLockProp->getFirstChild();
    if(pValNode){
      if(XNS(DOMNode)::ELEMENT_NODE == pValNode->getNodeType()){
	const XMLCh* pszType = pValNode->getLocalName();
	if(XNS(XMLString)::compareString(pszType, X("write")) == 0){
	  pLock->m_enuType = T_LOCK_WRITE;
	  pLock->m_unMask |= OI_LIVF_TYPE;
	}
      }
    }
  }

  //getting the <DAV:depth> element
  pLockPropList =
    ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("depth"));
  pLockProp = pLockPropList->item(0);
  if(pLockProp){
    //getting the text content of the node
    pValNode = pLockProp->getFirstChild();
    if(pValNode){
      if(XNS(DOMNode)::TEXT_NODE == pValNode->getNodeType()){
	const XMLCh* pszDepth = pValNode->getNodeValue();
	if(XNS(XMLString)::compareString(pszDepth, X("0")) == 0){
	  pLock->m_enuDepth = D_LOCK_ZERO;
	  pLock->m_unMask |= OI_LIVF_DEPTH;
	} else if(XNS(XMLString)::compareIString(pszDepth, X("infinity")) == 0){
	  pLock->m_enuDepth = D_LOCK_INFINITE;
	  pLock->m_unMask |= OI_LIVF_DEPTH;
	}
      }
    }
  }

  //getting the <DAV:timeout> element
  pLockPropList =
    ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("timeout"));
  pLockProp = pLockPropList->item(0);
  if(pLockProp){
    //getting the text content of the node
    pValNode = pLockProp->getFirstChild();
    if(pValNode){
      if(XNS(DOMNode)::TEXT_NODE == pValNode->getNodeType()){
	const XMLCh* pszTimeout = pValNode->getNodeValue();
	if(XNS(XMLString)::startsWith(pszTimeout, X("Second-"))){
	  pszTimeout += XNS(XMLString)::stringLen ("Second-");
	  pLock->m_lTimeout = atol(X(pszTimeout));
	  pLock->m_unMask |= OI_LIVF_TIMEOUT;
	} else if(XNS(XMLString)::compareIString(pszTimeout,X("Infinite")) == 0){
	  pLock->m_lTimeout = 0;
	  pLock->m_unMask |= OI_LIVF_TIMEOUT;
	} else {
	  pLock->m_lTimeout = 0;
	}
      }
    }
  }
 
  //getting the <DAV:owner> element, it's up to client program to parse the content
  pLockPropList =
    ((XNS(DOMElement)*)pEnum)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("owner"));
  pLockProp = pLockPropList->item(0);
  if(pLockProp) {
    pLock->m_pOwner = pLockProp;
    pLock->m_unMask |= OI_LIVF_OWNER;
  }

  if(pLock->m_unMask & OI_LIVF_ALL)
    return true;
  else
    return false;
}

void
CDavResourceNode::parseQuota(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("quota-bytes"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  XNS(DOMNode)* pQuota = pEnum->getFirstChild();
  if(pQuota){
    if(XNS(DOMNode)::TEXT_NODE == pQuota->getNodeType()){
      const XMLCh* pszQuota = pQuota->getNodeValue();
      if(pszQuota){
	X strQuota(pszQuota);
	char* pchPos = NULL;
	m_unQuota = OI_STRTOUL((const char*)strQuota, &pchPos, 10);
	m_ulValidProps |= OI_RNP_QUOTA;
      }
    }
  }
  pProp->removeChild(pEnum);
}

void
CDavResourceNode::parseQuotaUsed(XNS(DOMNode)* pProp)
{
  XNS(DOMNode)* pEnum;
  XNS(DOMNodeList)* pList =
    ((XNS(DOMElement)*)pProp)->getElementsByTagNameNS(X(OI_XML_DAVNAMESPACE),
						      X("quota-used-bytes"));
  pEnum = pList->item(0);
  if(!pEnum) 
    return;

  XNS(DOMNode)* pQuotaUsed = pEnum->getFirstChild();
  if(pQuotaUsed){
    if(XNS(DOMNode)::TEXT_NODE == pQuotaUsed->getNodeType()){
      const XMLCh* pszQuotaUsed = pQuotaUsed->getNodeValue();
      if(pszQuotaUsed){
	X strQuotaUsed(pszQuotaUsed);
	char* pchPos = NULL;
	m_unQuotaUsed = OI_STRTOUL((const char*)strQuotaUsed, &pchPos, 10);
	m_ulValidProps |= OI_RNP_QUOTAUSED;
      }
    }
  }
  pProp->removeChild(pEnum);
}


void
CDavResourceNode::removeUnprocessed(XNS(DOMNode)* pProp, int nStatus)
{
  if(!pProp)
    return;
  PROPSTATMAP::iterator it = m_mapPropStats.find(nStatus);
  if(it == m_mapPropStats.end()){
    m_mapPropStats[nStatus] = pProp;
  } else {
    XNS(DOMNode)* pOldProp = it->second;
    OI_ASSERT(pProp != pOldProp);
    for(XNS(DOMNode)* pUnprocessed = pProp->getFirstChild();
	pUnprocessed;
	pUnprocessed = pUnprocessed->getNextSibling()){
      pProp->removeChild(pUnprocessed);
      pOldProp->appendChild(pUnprocessed);
    }
  }
}
