/* 
   Http response body infomation container
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 1999-2004, Joe Orton <joe@manyfish.co.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavResponseBody.cpp 132 2005-06-24 09:09:43Z komat $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/DavResponseBody.h>
#include <onion/DavRequest.h>
#include <onion/DavSocket.h>
#include <onion/DavWorkSession.h>
#include <onion/HandlerTE.h>
#include <onion/HandlerCLength.h>

CDavResponseBody::CDavResponseBody(CDavRequest* pReq,
				   CDavSocket* pSocket):
  m_pReq(pReq),
  m_pSocket(pSocket)
{
  OI_ASSERT(pReq);
  OI_ASSERT(pSocket);

  m_pszLocalBuf = new char[OI_LINEBUFSIZE];

  //TE handler and CLength handler are related to response body
  m_pReq->AddHandler(new CHandlerTE(this));
  m_pReq->AddHandler(new CHandlerCLength(this));
#ifdef HAVE_LIBZ
  m_pReq->AddHandler(new CHandlerCEncoding(this));
  m_pZreader = NULL;
#endif
  Reset();
}

CDavResponseBody::~CDavResponseBody()
{
  delete[] m_pszLocalBuf;
#ifdef HAVE_LIBZ
  OI_ASSERT(m_pZreader);
  delete m_pZreader;
#endif
}

void
CDavResponseBody::Reset()
{
  m_enuLastErr	= OI_OK;
  m_enuMode = M_RSP_TILLEOF;
  m_bEOF = false;

  m_unBodyLeft = 0;
  m_unBodyLen = 0;
  m_unChunkLeft	= 0;
  m_unChunkLen = 0;
  m_unReadLen = 0;
  m_unChunkNo = 0;

#ifdef HAVE_LIBZ
  m_bCompressed = false;
  if(m_pZreader)
    delete m_pZreader;
  m_pZreader = new CDavZReader(this);
#endif
}

#define OI_TERMINATE_READ(enuRet) {m_enuLastErr = enuRet;return enuRet;}

OI_RESULT
CDavResponseBody::read(char* pszBuf, 
		       size_t unBufLen,
		       size_t* punReadLen)
{
  OI_RESULT enuRet;
  size_t unReadLen;
  OI_SIZE_T unWillRead;

  if(!unBufLen)
    return OIGEINVALIDARG;

  switch(m_enuMode){
  case M_RSP_CHUNKED:
    if(m_unChunkLeft == 0){
      OI_SIZE_T unChunkLen;
      char* pchPos = 0;
      unReadLen = OI_LINEBUFSIZE;
      enuRet = m_pSocket->ReadLine(m_pszLocalBuf, &unReadLen);
      if(OI_OK != enuRet){
	OI_TERMINATE_READ(enuRet);
      }
      if(unReadLen >= 11){
	OI_TERMINATE_READ(OIHECHUNKSIZE);
      }
      unChunkLen = strtoul(m_pszLocalBuf, &pchPos, 16);
      if(pchPos == m_pszLocalBuf || unChunkLen == ULONG_MAX){
	OI_TERMINATE_READ(OIHECHUNKSIZE);
      }
      if(unChunkLen == 0){
	//end of body reached
	if(punReadLen)
	  *punReadLen = 0;
	return OI_OK;
      }
      m_unChunkLeft = unChunkLen;
    }
    unWillRead = m_unChunkLeft;
    break;
  case M_RSP_CLENGTH:
      unWillRead = m_unBodyLeft;
      break;
  case M_RSP_TILLEOF:
    unWillRead = unBufLen;
    break;
  case M_RSP_NO_BODY:
  default:
    unWillRead = 0;
    break;
  }

  if(unWillRead > unBufLen){
    unWillRead = unBufLen;
  } else if(unWillRead == 0) {
      if(punReadLen)
	*punReadLen = 0;
      return OI_OK;
  }

  unReadLen = (size_t)unWillRead;
  enuRet = m_pSocket->Read(pszBuf, &unReadLen);

  if((m_enuMode == M_RSP_TILLEOF) &&
     ((enuRet == OISECLOSED) ||
      ((enuRet == OISETRUNC) && (m_unBodyLen == 0)))){
    m_pReq->SetPersistent(false);
    unReadLen = 0;
    if(punReadLen)
      *punReadLen = 0;
    return OI_OK;
  } else if(OI_OK != enuRet) {
    OI_TERMINATE_READ(enuRet);
  } else {
    //read successful
  }

  if(punReadLen)
    *punReadLen = unReadLen;
  m_unReadLen += unReadLen;

  if(m_enuMode == M_RSP_CHUNKED){
    m_unChunkLeft -= unReadLen;
    if(m_unChunkLeft == 0){
      //check if chunk is ended with CRLF
      unReadLen = 2;
      enuRet = m_pSocket->FullRead(m_pszLocalBuf, &unReadLen);
      if(OI_OK != enuRet) 
	OI_TERMINATE_READ(enuRet);

      if(strncmp(m_pszLocalBuf, "\r\n", 2) != 0)
	OI_TERMINATE_READ(OIHEPROTOCOLPANIC);
    }
  } else if(m_enuMode == M_RSP_CLENGTH) {
    m_unBodyLeft -= unReadLen;
  }

  return OI_RETRY;
}
#undef OI_TERMINATE_READ

size_t
CDavResponseBody::read(unsigned char* toFill, size_t maxToRead)
{
  size_t unReadLen;
  OI_RESULT enuRet;
  CDavWorkSession* pSession = m_pReq->GetSession();
  OI_ASSERT(pSession);

  enuRet = this->Read((char*)toFill, maxToRead, &unReadLen);
  if(OI_RETRY != enuRet){
    m_enuLastErr = enuRet;
    unReadLen = 0;
  } else {
    bool bContinue =
      pSession->OnReceiveProgress(m_pReq,
				  m_unReadLen,
				  m_unBodyLen,
				  (m_enuMode==M_RSP_CLENGTH));
    
    if(!bContinue){
      //TODO: better cancel handling
      m_enuLastErr = OI_USERCANCELED;
      pSession->Disconnect();
      unReadLen = 0;
    }
  }

  return unReadLen;
}

OI_RESULT
CDavResponseBody::Read(char* pszBuf, size_t unBufLen, size_t* punReadLen)
{
  OI_RESULT enuRet;
  size_t unReadLen = 0;
#ifdef HAVE_LIBZ
  if(m_bCompressed)
    enuRet = m_pZreader->Read(pszBuf, unBufLen, &unReadLen);
  else
#endif
    enuRet = read(pszBuf, unBufLen, &unReadLen);
  m_pReq->GetSession()->m_unResponseLength += unReadLen;
  if(punReadLen)
    *punReadLen = unReadLen;
  m_bEOF = (OI_OK == enuRet);
  return enuRet;
}

size_t
CDavResponseBody::Read(unsigned char* toFill, size_t maxToRead)
{
  size_t unReadLen = read(toFill, maxToRead);
  m_pReq->GetSession()->m_unResponseLength += unReadLen;
  m_bEOF = (unReadLen == 0);
  return unReadLen;
}

#ifdef HAVE_LIBZ
#define OI_MINZBUFSIZE		10
#define OI_ZLIB_RESERVED_BITS	0xe0
#define OI_ZLIB_HEADER_CRC	0x02
#define OI_ZLIB_EXTRA_FIELDS	0x04
#define OI_ZLIB_ORIG_NAME	0x08
#define OI_ZLIB_COMMENT		0x10

CDavZReader::CDavZReader(CDavResponseBody* pBody)
  :m_pBody(pBody)
{
  OI_ASSERT(pBody);
  m_unBufferSize = OI_GENBUFSIZE;
  m_bInitialized = false;
  m_bEOF = true;
  m_bStreamEnd = true;

  m_pBuffer = (OI_BYTE*)malloc(sizeof(OI_BYTE) * m_unBufferSize);
}

CDavZReader::~CDavZReader()
{
  if(m_bInitialized)
    inflateEnd(&m_Zstream);
  free(m_pBuffer);
}

size_t
CDavZReader::SetBufferSize(size_t unNewSize)
{
  if(unNewSize > OI_MINZBUFSIZE) {
    m_pBuffer = (OI_BYTE*)realloc((void*)m_pBuffer, unNewSize);
    if(m_pBuffer) {
      m_unBufferSize = unNewSize;
    } else {
      m_unBufferSize = 0;
    }
  }
  return m_unBufferSize;
}

OI_RESULT
CDavZReader::parseHeader()
{
  OI_RESULT enuResult;
  OI_BYTE yHeader[10];
  enuResult = readBody(yHeader, 10, NULL);
  if(OI_RETRY != enuResult)
    return OIZEBADHEADER;

  m_stHeader.ID1	= yHeader[0];
  m_stHeader.ID2	= yHeader[1];
  m_stHeader.CM		= yHeader[2];
  m_stHeader.FLAG	= yHeader[3];
  m_stHeader.MTIME	= yHeader[4];
  m_stHeader.MTIME	+= yHeader[5] << 8;
  m_stHeader.MTIME	+= yHeader[6] << 16;
  m_stHeader.MTIME	+= yHeader[7] << 24;
  m_stHeader.XFL	= yHeader[8];
  m_stHeader.OS		= yHeader[9];

  /* hard coded header signature */
  if(m_stHeader.ID1 != 0x1f || m_stHeader.ID2 != 0x8b)
    return OIZEBADHEADER;

  if(m_stHeader.CM != Z_DEFLATED || 
     (m_stHeader.FLAG & OI_ZLIB_RESERVED_BITS))
    return OIZEBADHEADER;
  /* discard extra fields */
  if(m_stHeader.FLAG & OI_ZLIB_EXTRA_FIELDS) {
    OI_BYTE yExtraFieldLen[2];
    enuResult = readBody(yExtraFieldLen, 2, NULL);
    if(OI_RETRY != enuResult)
      return OIZEBADHEADER;

    size_t unExtraLen = yExtraFieldLen[0] + (yExtraFieldLen[2] << 8);
    size_t unIndex;
    for(unIndex = 0; unIndex < unExtraLen; unIndex++) {
      enuResult = readBody(yExtraFieldLen, 1, NULL);
      if(OI_RETRY != enuResult)
	return OIZEBADHEADER;
    }
  }

  /* discard orig name */
  if(m_stHeader.FLAG & OI_ZLIB_ORIG_NAME) {
    OI_BYTE yNameBuf[1] = {0xff};
    char chCur;
    for(; chCur = (char)yNameBuf[0];) {
      enuResult = readBody(yNameBuf, 1, NULL);
      if(OI_RETRY != enuResult)
	return OIZEBADHEADER;
    }
  }

  /* discard comment */
  if(m_stHeader.FLAG & OI_ZLIB_COMMENT) {
    OI_BYTE yCommentBuf[1] = {0xff};
    char chCur;
    for(; chCur = (char)yCommentBuf[0];) {
      enuResult = readBody(yCommentBuf, 1, NULL);
      if(OI_RETRY != enuResult)
	return OIZEBADHEADER;
    }
  }

  /* discard CRC */
  if(m_stHeader.FLAG & OI_ZLIB_HEADER_CRC) {
    OI_BYTE yCRCBuf[2];
    enuResult = readBody(yCRCBuf, 2, NULL);
    if(OI_RETRY != enuResult)
      return OIZEBADHEADER;
  }

  return OI_OK;
}

OI_RESULT
CDavZReader::readBody(OI_BYTE* pBuf, size_t unBufLen, size_t* punReadLen)
{
  OI_ASSERT(pBuf);
  OI_RESULT enuResult;
  size_t unReadLen = 0;
  for(;unReadLen < unBufLen;) {
    size_t unRead = 0;
    size_t unToRead = unBufLen - unReadLen;
    enuResult = m_pBody->read((char*)(pBuf + unReadLen), unToRead, &unRead);
    unReadLen += unRead;
    if(OI_RETRY != enuResult)
      break;
  }
  if(punReadLen)
    *punReadLen = unReadLen;
  return enuResult;
}

OI_RESULT
CDavZReader::Read(char* pszBuf, size_t unBufLen, size_t* punReadLen)
{
  OI_RESULT enuRet;
  if(!m_bInitialized) {
    enuRet = init();
    if(OI_OK != enuRet)
      return enuRet;
  }

  if(m_bStreamEnd)
    return OI_OK;

  OI_ASSERT(m_bInitialized);
  m_Zstream.next_out = (Bytef*)pszBuf;
  m_Zstream.avail_out = unBufLen;

  size_t unReadLen;
  int nErr = Z_OK;
  for(; m_Zstream.avail_out != 0 && nErr != Z_STREAM_END;) {
    if((m_Zstream.avail_in == 0) && !m_bEOF) {
      /* read data in */
      enuRet = m_pBody->read((char*)m_pBuffer, m_unBufferSize, &unReadLen);
      if(OI_RETRY != enuRet) {
	if(OI_OK != enuRet)
	  return enuRet;
	else {
	  /*
	   * FIXME: server = apache2 + mod_deflate
	   * while using ranged get with compression, end of zstream
	   * and footer is not sent. Is this a bug?
	   */
	  m_bEOF = true;
	  return OIZEBADDATA;
	}
      }
      m_Zstream.next_in = m_pBuffer;
      m_Zstream.avail_in = unReadLen;
    }
    nErr = inflate(&m_Zstream, Z_SYNC_FLUSH);
    if(nErr == Z_STREAM_END) {
      m_bStreamEnd = true;
      m_ulChecksum = crc32(m_ulChecksum, (Bytef*)pszBuf,
			   m_Zstream.next_out - (Bytef*)pszBuf);
      if(m_Zstream.avail_in > 8)
	return OIZEBADDATA;

      OI_BYTE yFooter[8];
      size_t unDelta = 8 - m_Zstream.avail_in;
      memcpy(yFooter, m_Zstream.next_in, m_Zstream.avail_in);

      if(unDelta) {
	size_t unLen;
	enuRet = readBody(yFooter + m_Zstream.avail_in, unDelta, &unLen);
	if(OI_OK != enuRet)
	  return OIZEBADDATA;
      }
      uLong ulCRC32;
      ulCRC32 = yFooter[0];
      ulCRC32 += yFooter[1] << 8;
      ulCRC32 += yFooter[2] << 16;
      ulCRC32 += yFooter[3] << 24;

      if(ulCRC32 != m_ulChecksum)
	return OIZEBADDATA;
      else {
	uLong ulSize;
	ulSize = yFooter[4];
	ulSize += yFooter[5] << 8;
	ulSize += yFooter[6] << 16;
	ulSize += yFooter[7] << 24;
	/*TODO: remove this for large file transfer*/
	if(ulSize != m_Zstream.total_out)
	  return  OIZEBADDATA;
      }

      if(punReadLen)
	*punReadLen = unBufLen - m_Zstream.avail_out;
      return OI_RETRY;
      //return OI_OK;???
    } else if(nErr != Z_OK) {
      return OIZEBADDATA;
    }
  }
  m_ulChecksum = crc32(m_ulChecksum, (Bytef*)pszBuf, m_Zstream.next_out - (Bytef*)pszBuf);
  if(punReadLen)
    *punReadLen = unBufLen - m_Zstream.avail_out;
  return OI_RETRY;
}

OI_RESULT
CDavZReader::init()
{
  OI_RESULT enuRet = parseHeader();
  if(OI_OK != enuRet)
    return enuRet;

  m_Zstream.zalloc	= Z_NULL;
  m_Zstream.zfree	= Z_NULL;
  m_Zstream.opaque	= Z_NULL;
  m_Zstream.next_in	= m_pBuffer;
  m_Zstream.avail_in	= 0;

  /*TODO: try other settings?*/
  int nErr = inflateInit2(&m_Zstream, -MAX_WBITS);
  if(Z_OK != nErr)
    return OIZEINITFAILED;

  m_ulChecksum = crc32(0, Z_NULL, 0);
  m_bInitialized = true;
  m_bEOF = false;
  m_bStreamEnd = false;
  return OI_OK;
}

CHandlerCEncoding::CHandlerCEncoding(CDavResponseBody* pBody)
  :m_pBody(pBody)
{
  OI_ASSERT(pBody);
}

CHandlerCEncoding::~CHandlerCEncoding()
{
}

const char*
CHandlerCEncoding::GetHeaderName()
{
  return OI_RSPHDR_CENCODING;
}

OI_RESULT 
CHandlerCEncoding::Execute(CDavRequest* pReq, OI_STRLIST_A& vecValues)
{
  OI_ASSERT(vecValues.size());
  OI_STRING_A strValue(vecValues[0]), strGzip("gzip");
  OI_STRING_A::size_type nPos = strValue.find(strGzip);
  if(nPos != OI_STRING_A::npos) {
    m_pBody->m_bCompressed = true;
  } else {
    m_pBody->m_bCompressed = false;
  }
  return OI_OK;
}

#endif /*HAVE_LIBZ*/
