/* 
   SSL certificate
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 2002-2004, Joe Orton <joe@manyfish.co.uk>
   Copyright (C) 1999-2000 Tommi Komulainen <Tommi.Komulainen@iki.fi>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavSSLCertificate.cpp 132 2005-06-24 09:09:43Z komat $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <onion/DavSSLCertificate.h>
#include <openssl/asn1.h>
#include <openssl/bn.h>
#include <openssl/evp.h>
#include <openssl/x509v3.h>

static void
bn2string(BIGNUM* pBN, OI_STRING_A& strOut)
{
  if(!pBN)
    return;
  strOut.erase();

  char* pBNStr = BN_bn2hex(pBN);
  if(pBNStr) {
    strOut = pBNStr;
    OPENSSL_free(pBNStr);
  }
}

static void
asn1int2string(ASN1_INTEGER* pInt, OI_STRING_A& strOut)
{
  BIGNUM* pBN = BN_new();
  if(ASN1_INTEGER_to_BN(pInt, pBN)) {
    bn2string(pBN, strOut);
  }
  BN_free(pBN);
}

static void 
asn1TimeToString(ASN1_TIME* pTime, OI_STRING_A& strOut)
{
  char chBuf[64];
  BIO *pBIO;
	
  strncpy(chBuf, "[invalid date]", sizeof chBuf);
  pBIO = BIO_new(BIO_s_mem());
  if (pBIO){
    if (ASN1_TIME_print(pBIO, pTime))
      BIO_read(pBIO, chBuf, sizeof chBuf);
    BIO_free(pBIO);
  }
  strOut = chBuf;
}

CDavSSLCertificate::CDavSSLCertificate()
{
  m_pX509 = NULL;
  m_ulFailures = 0L;
}

CDavSSLCertificate::CDavSSLCertificate(const CDavSSLCertificate& master)
{
  if(!master.m_pX509)
    return;

  CRYPTO_add(&master.m_pX509->references, 1, CRYPTO_LOCK_X509);
  m_pX509 = master.m_pX509;

  m_ulFailures = master.m_ulFailures;
  return;
}

CDavSSLCertificate::CDavSSLCertificate(X509* pX509, unsigned long ulFailures)
{
  OI_ASSERT(pX509);
  
  CRYPTO_add(&pX509->references, 1, CRYPTO_LOCK_X509);
  m_pX509 = pX509;

  m_ulFailures = ulFailures;
}

CDavSSLCertificate::~CDavSSLCertificate()
{
  if(m_pX509)
    X509_free(m_pX509);
}

void 
CDavSSLCertificate::Attach(X509* pX509)
{
  if(m_pX509)
    X509_free(m_pX509);
  m_pX509 = pX509;
}

X509* 
CDavSSLCertificate::Detach()
{
  X509* pX509 = m_pX509;
  m_pX509 = NULL;
  return pX509;
}

CDavSSLCertificate& 
CDavSSLCertificate::operator =(const CDavSSLCertificate& master)
{
  if(m_pX509){
    X509_free(m_pX509);
    m_pX509 = NULL;
    m_ulFailures = 0;
  }

  if(!master.m_pX509)
    return *this;

  CRYPTO_add(&master.m_pX509->references, 1, CRYPTO_LOCK_X509);
  m_pX509 = master.m_pX509;

  m_ulFailures = master.m_ulFailures;
  return *this;
}

CDavSSLCertificate& 
CDavSSLCertificate::operator =(X509* pX509)
{
  if(m_pX509){
    X509_free(m_pX509);
    m_pX509 = NULL;
    m_ulFailures = 0;
  }

  if(!pX509)
    return *this;

  CRYPTO_add(&pX509->references, 1, CRYPTO_LOCK_X509);
  m_pX509 = pX509;

  return *this;
}

bool 
CDavSSLCertificate::operator ==(const CDavSSLCertificate& other)
{
  if(!m_pX509 || !other.m_pX509)
    return false;
  else
    return (X509_cmp(m_pX509, other.m_pX509) == 0);
}

bool 
CDavSSLCertificate::operator ==(const X509* pOther)
{
  if(!m_pX509 || !pOther)
    return false;
  else
    return (X509_cmp(m_pX509, pOther) == 0);
}

X509*
CDavSSLCertificate::GetX509()
{
  return m_pX509;
}

static void
loadCB(X509* pX509, void* pvUserData)
{
  CDavSSLCertificate* pCert = (CDavSSLCertificate*)pvUserData;
  *pCert = pX509;
}

bool
CDavSSLCertificate::Load(BIO* pBIO,
			 OI_SSL_CRTENC enuEncoding)
{
  if(!pBIO)
    return false;
  int nCertCount = ReadCertificates(pBIO, MSG_CRT_X509, enuEncoding,
				    loadCB, (void*)this);
  return (nCertCount == 1);
}

bool
CDavSSLCertificate::Load(const char* pszPath,
			 OI_SSL_CRTENC enuEncoding)
{
  if(!pszPath)
    return false;
  BIO* pBIO = BIO_new(BIO_s_file());
  if(!BIO_read_filename(pBIO, pszPath))
    return false;
  bool bRet = Load(pBIO, enuEncoding);
  BIO_free(pBIO);
  return bRet;
}

bool
CDavSSLCertificate::Load(void* pvData,
			 unsigned int unLen,
			 OI_SSL_CRTENC enuEncoding)
{
  if(!pvData || !unLen)
    return false;
  BIO* pBIO = BIO_new_mem_buf(pvData, unLen);
  if(!pBIO)
    return false;
  bool bRet = Load(pBIO, enuEncoding);
  BIO_free(pBIO);
  return bRet;
}

void
CDavSSLCertificate::Clear()
{
  if(m_pX509)
    X509_free(m_pX509);
  m_pX509 = NULL;
}

void
CDavSSLCertificate::Save(const char* pszPath,
			 OI_SSL_CRTENC enuEncoding)
{
  if(!m_pX509 || !pszPath)
    return;
  BIO* pBIO = BIO_new_file(pszPath, "w");
  switch(enuEncoding) {
  case ENC_CRT_DER:
    i2d_X509_bio(pBIO, m_pX509);
    break;
  case ENC_CRT_PEM:
  default:
    PEM_write_bio_X509(pBIO, m_pX509);
    break;
  }
  BIO_free(pBIO);
}

bool
CDavSSLCertificate::Export(CDavSSLCertificateData& dataOut,
			   OI_SSL_CRTENC enuEncoding)
{
  if(!m_pX509)
    return false;

  dataOut.Clear();
  BIO* pBIO = dataOut.GetBIO();
  if(!pBIO)
    return false;

  int nRet = -1;
  switch(enuEncoding) {
  case ENC_CRT_PEM:
    nRet = PEM_write_bio_X509(pBIO, m_pX509);
    break;
  case ENC_CRT_DER:
  default:
    nRet = i2d_X509_bio(pBIO, m_pX509);
    break;
  }
  if(nRet < 0) {
    return false;
  } else {
    BIO_set_flags(pBIO, BIO_FLAGS_MEM_RDONLY);
    BIO_set_mem_eof_return(pBIO, 0);
  }
  return true;
}

bool
CDavSSLCertificate::GetIssuer(CDavSSLDName& dnOut)
{
  if(!m_pX509)
    return false;

  X509_NAME* pIssuer = X509_get_issuer_name(m_pX509);
  return dnOut.ParseDName(pIssuer);
}

bool
CDavSSLCertificate::GetSubject(CDavSSLDName& dnOut)
{
  if(!m_pX509)
    return false;

  X509_NAME* pSubject = X509_get_subject_name(m_pX509);
  return dnOut.ParseDName(pSubject);
}

time_t
CDavSSLCertificate::GetNotBefore()
{
  if(!m_pX509)
    return -1;

  ASN1_TIME *pNotBefore = X509_get_notBefore(m_pX509);
  if(!pNotBefore)
    return -1;
  OI_STRING_A strNotBefore;
  asn1TimeToString(pNotBefore, strNotBefore);
  return ParseTimeRFC2459(strNotBefore.c_str());
}

time_t
CDavSSLCertificate::GetNotAfter()
{
  if(!m_pX509)
    return -1;

  ASN1_TIME *pNotAfter = X509_get_notAfter(m_pX509);
  if(!pNotAfter)
    return -1;
  OI_STRING_A strNotAfter;
  asn1TimeToString(pNotAfter, strNotAfter);
  return ParseTimeRFC2459(strNotAfter.c_str());
}

bool
CDavSSLCertificate::GetSerialNumber(OI_STRING_A& strOut)
{
  if(!m_pX509)
    return false;
  ASN1_INTEGER* pInt = X509_get_serialNumber(m_pX509);
  if(!pInt)
    return false;

  asn1int2string(pInt, strOut);
  return true;
}

bool
CDavSSLCertificate::GetPublicKey(OI_STRING_A& strOut)
{
  if(!m_pX509)
    return false;

  EVP_PKEY* pKey = X509_get_pubkey(m_pX509);
  if(!pKey)
    return false;

  switch(pKey->type) {
  case EVP_PKEY_DSA:
    bn2string(pKey->pkey.dsa->pub_key, strOut);
    break;
  case EVP_PKEY_RSA:
    bn2string(pKey->pkey.rsa->n, strOut);
    break;
  default:
    strOut = "Algorithm not supported";
    break;
  }
  return true;
}

/*
  EVP_PKEY_type() returns the type of key corresponding to the value
  type. The type of a key can be obtained with EVP_PKEY_type(pkey->type).
  The return value will be EVP_PKEY_RSA, EVP_PKEY_DSA, EVP_PKEY_DH or
  ????EVP_PKEY_EC???? for the corresponding key types or NID_undef if 
  the key type is unassigned.
 */
bool
CDavSSLCertificate::GetPublicKeyInfo(OIPKEYINFO *pKeyInfo)
{
  if(!m_pX509 || !pKeyInfo)
    return false;
  EVP_PKEY* pKey = X509_get_pubkey(m_pX509);
  if(!pKey)
    return false;

  switch(pKey->type) {
  case EVP_PKEY_NONE:
    pKeyInfo->enuAlgorithm = ALG_PKEY_NONE;
    pKeyInfo->nKeyLength = -1;
    break;
  case EVP_PKEY_RSA:
    pKeyInfo->enuAlgorithm = ALG_PKEY_RSA;
    pKeyInfo->nKeyLength = BN_num_bits(pKey->pkey.rsa->n);
    break;
  case EVP_PKEY_DSA:
    pKeyInfo->enuAlgorithm = ALG_PKEY_DSA;
    pKeyInfo->nKeyLength = BN_num_bits(pKey->pkey.dsa->p);
    break;
  default:
    return false;
  }
  return true;
}

bool
CDavSSLCertificate::GetSignature(OI_STRING_A& strOut)
{
  if(!m_pX509)
    return false;

  ASN1_BIT_STRING* pSignature = m_pX509->signature;
  if(!pSignature)
    return false;
  asn1int2string(pSignature, strOut);
  return true;
}

bool
CDavSSLCertificate::GetSignatureAlgorithm(OI_SSL_PKEYALG& algOut)
{
  if(!m_pX509)
    return false;

  int nNID = X509_get_signature_type(m_pX509);
  switch(nNID) {
  case EVP_PKEY_DSA:
    algOut = ALG_PKEY_DSA;
    break;
  case EVP_PKEY_RSA:
    algOut = ALG_PKEY_RSA;
    break;
  case EVP_PKEY_DH:
    algOut = ALG_PKEY_DH;
    break;
  case NID_undef:
  default:
    algOut = ALG_PKEY_NONE;
    break;
  }

  return true;
}

unsigned long
CDavSSLCertificate::GetSubjectHash()
{
  if(!m_pX509)
    return 0;
  return X509_NAME_hash(m_pX509->cert_info->subject);
}

long
CDavSSLCertificate::GetVersion()
{
  if(!m_pX509)
    return -1;
  else
    return X509_get_version(m_pX509);
}

bool
CDavSSLCertificate::IsIssuedBy(CDavSSLDName& dnIssuer)
{
  if(!m_pX509)
    return false;
  const X509_NAME* pIssuer = X509_get_issuer_name(m_pX509);
  if(!pIssuer)
    return false;
  return (dnIssuer == pIssuer);
}

bool
CDavSSLCertificate::IsSelfSigned()
{
  if(!m_pX509)
    return false;
  return (X509_check_issued(m_pX509, m_pX509) == X509_V_OK);
}

int
CDavSSLCertificate::Compare(CDavSSLCertificate& crtOther)
{
  if(!m_pX509)
    return -1;
  if(!crtOther.m_pX509)
    return 1;
  return X509_cmp(m_pX509, crtOther.m_pX509);
}

int
CDavSSLCertificate::CompareSubject(CDavSSLCertificate& crtOther)
{
  if(!m_pX509)
    return -1;
  if(!crtOther.m_pX509)
    return 1;

  X509_NAME* pName = X509_get_subject_name(m_pX509);
  X509_NAME* pOtherName = X509_get_subject_name(crtOther.m_pX509);
  if(!pName)
    return -1;
  if(!pOtherName)
    return 1;
  return X509_NAME_cmp(pName, pOtherName);
}
