/* 
   List of SSL certificates
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavSSLCertificateList.cpp 132 2005-06-24 09:09:43Z komat $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/DavSSLCertificateList.h>
#include <onion/DavSSLCertificate.h>
#include <openssl/x509.h>
#include <openssl/pkcs7.h>

static void
loadCertCB(X509* pX509, void* pvUserData)
{
  OI_ASSERT(pX509 && pvUserData);
  CDavSSLCertificateList* pList = (CDavSSLCertificateList*)pvUserData;
  pList->AddCertificate(CDavSSLCertificate(pX509, 0));
}

CDavSSLCertificateList::CDavSSLCertificateList()
{
}

CDavSSLCertificateList::~CDavSSLCertificateList()
{
}

unsigned int 
CDavSSLCertificateList::GetCount()
{
  return m_vecCertificateList.size();
}

void 
CDavSSLCertificateList::Clear()
{
  m_vecCertificateList.clear();
}

void 
CDavSSLCertificateList::AddCertificate(const CDavSSLCertificate& cert)
{
  m_vecCertificateList.push_back(cert);
}

CDavSSLCertificate& 
CDavSSLCertificateList::operator [](unsigned int unIndex)
{
  return m_vecCertificateList[unIndex];
}

int
CDavSSLCertificateList::Load(BIO* pBIO,
			     OI_SSL_CRTMSG enuMsgFormat,
			     OI_SSL_CRTENC enuEncFormat)
{
  if(!pBIO)
    return -1;
  return ReadCertificates(pBIO, enuMsgFormat, enuEncFormat,
			  loadCertCB, (void*)this);
}

int
CDavSSLCertificateList::Load(const char* pszPath,
			     OI_SSL_CRTMSG enuMsgFormat,
			     OI_SSL_CRTENC enuEncFormat)
{
  if(!pszPath)
    return -1;
  BIO* pBIO = BIO_new(BIO_s_file());
  if(!BIO_read_filename(pBIO, pszPath))
    return -1;
  int nRet =  Load(pBIO, enuMsgFormat, enuEncFormat);
  BIO_free(pBIO);
  return nRet;
}

int
CDavSSLCertificateList::Load(void* pvData,
			     unsigned int unLen,
			     OI_SSL_CRTMSG enuMsgFormat,
			     OI_SSL_CRTENC enuEncFormat)
{
  if(!pvData || !unLen)
    return -1;
  BIO* pBIO = BIO_new_mem_buf(pvData, unLen);
  if(!pBIO)
    return -1;
  int nRet =  Load(pBIO, enuMsgFormat, enuEncFormat);
  BIO_free(pBIO);
  return nRet;
}

bool
CDavSSLCertificateList::Save(const char* pszPath,
			     OI_SSL_CRTENC enuEncFormat)
{
  unsigned int unCount = GetCount();
  if(!unCount || !pszPath)
    return false;

  int nRet = -1;
  unsigned int unIndex;
  PKCS7* pPKCS7 = PKCS7_new();
  if(!pPKCS7)
    return false;

  do{
    PKCS7_SIGNED* pP7S = PKCS7_SIGNED_new();
    if(!pP7S)
      break;
    pPKCS7->type=OBJ_nid2obj(NID_pkcs7_signed);
    pPKCS7->d.sign=pP7S;
    pP7S->contents->type=OBJ_nid2obj(NID_pkcs7_data);
    if (!ASN1_INTEGER_set(pP7S->version,1))
      break;
    STACK_OF(X509)* pStack = sk_X509_new_null();
    if (!pStack)
      break;
    pP7S->cert=pStack;


    for(unIndex = 0; unIndex < unCount; unIndex++) {
      CDavSSLCertificate& cert =m_vecCertificateList[unIndex];
       X509* pX509 = cert.GetX509();
      if(!pX509)
	break;
      sk_X509_push(pStack, pX509);
      CRYPTO_add(&pX509->references,1,CRYPTO_LOCK_X509);
    }
    BIO* pBIO = BIO_new_file(pszPath, "w");
    if(pBIO) {
      switch(enuEncFormat) {
      case ENC_CRT_DER:
	nRet = i2d_PKCS7_bio(pBIO, pPKCS7);
	break;
      case ENC_CRT_PEM:
      default:
	nRet = PEM_write_bio_PKCS7(pBIO, pPKCS7);
	break;
      }
      BIO_free(pBIO);
    }

  } while(0);

  PKCS7_free(pPKCS7);
  return (nRet >= 0);
}
