/* 
   SSL client certificate
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavSSLClientCertificate.cpp 132 2005-06-24 09:09:43Z komat $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/DavSSLClientCertificate.h>
#include <openssl/pkcs12.h>
//#include <openssl/bio.h>

CDavSSLClientCertificate::CDavSSLClientCertificate()
{
  m_pKey = NULL;
}

CDavSSLClientCertificate::~CDavSSLClientCertificate()
{
  if(m_pKey)
    EVP_PKEY_free(m_pKey);
}

CDavSSLClientCertificate&
CDavSSLClientCertificate::operator =(CDavSSLClientCertificate& master)
{
  if(master.IsValid()) {
    Clear();
    m_crtMain = master.m_crtMain;
    CRYPTO_add(&master.m_pKey->references, 1, CRYPTO_LOCK_EVP_PKEY);
    m_pKey = master.m_pKey;
  }
  return *this;
}

bool
CDavSSLClientCertificate::IsValid()
{
  return (m_crtMain.GetX509() && m_pKey);
}

static bool
loadClientCert(BIO* pBIO,
	       const char* pszPasswd,
	       CDavSSLCertificate& crtOut,
	       EVP_PKEY** ppKey)
{
  OI_ASSERT(pBIO && pszPasswd && ppKey);
  bool bRet = false;
  PKCS12* pP12 = d2i_PKCS12_bio(pBIO, NULL);
  if(pP12) {
    X509* pX509;
    if(PKCS12_parse(pP12, pszPasswd, ppKey, &pX509, NULL) == 1) {
      OI_ASSERT(pX509 && ppKey);
      bRet = true;
      crtOut = pX509;
      X509_free(pX509);
    }
    PKCS12_free(pP12);
  }
  return bRet;
}

bool
CDavSSLClientCertificate::Load(const char* pszPath,
			       const char* pszPasswd)
{
  if(!pszPath || !pszPasswd)
    return false;
  Clear();
  bool bRet = false;
  BIO* pBIO = BIO_new_file(pszPath, "rb");
  if(pBIO) {
    bRet = loadClientCert(pBIO, pszPasswd,
			  m_crtMain, &m_pKey);
    BIO_free(pBIO);
  }
  return bRet;
}

bool
CDavSSLClientCertificate::Load(void* pvData,
			       unsigned int unLen,
			       const char* pszPasswd)
{
  if(!pvData || !unLen || !pszPasswd)
    return false;
  Clear();
  bool bRet = false;
  BIO* pBIO = BIO_new_mem_buf(pvData, unLen);
  if(pBIO) {
    bRet = loadClientCert(pBIO, pszPasswd,
			  m_crtMain, &m_pKey);
    BIO_free(pBIO);
  }
  return bRet;
}

void
CDavSSLClientCertificate::Clear()
{
  if(m_crtMain.GetX509())
    m_crtMain.Clear();
  if(m_pKey) {
    EVP_PKEY_free(m_pKey);
    m_pKey = NULL;
  }
}

CDavSSLCertificate&
CDavSSLClientCertificate::Cert()
{
  return m_crtMain;
}

EVP_PKEY*
CDavSSLClientCertificate::Key()
{
  return m_pKey;
}
