/* 
   Distinguish name of SSL certificates
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavSSLDName.cpp 132 2005-06-24 09:09:43Z komat $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/DavSSLDName.h>


CDavSSLDName::CDavSSLDName()
{
  m_pX509Name = NULL;
  m_itrCurrent = m_mapFields.begin();
}

CDavSSLDName::CDavSSLDName(const X509_NAME* pName)
{
  OI_ASSERT(pName);
  m_pX509Name = (X509_NAME*)pName;
}

CDavSSLDName::~CDavSSLDName()
{
}

bool 
CDavSSLDName::ParseDName(X509_NAME* pName)
{
  //OI_ASSERT(pName);
  if(!pName)
    return false;
  m_mapFields.clear();
  m_pX509Name = pName;

  char chBuf[80];
  OI_STRING_A strName;
  OI_STRING_A strValue;
  X509_NAME_ENTRY* pEntry = NULL;
  ASN1_OBJECT* pObject = NULL;
  bool bDoIt[4] = {false, false, false, false};

  int nCount = sk_X509_NAME_ENTRY_num(pName->entries);

  for(int nIndex = 0; nIndex < nCount; nIndex++){
    pEntry = sk_X509_NAME_ENTRY_value(pName->entries, nIndex);
    pObject = X509_NAME_ENTRY_get_object(pEntry);
    int	nID = OBJ_obj2nid(pEntry->object);

    const char* pszName;
    if ((nID == NID_undef) || ((pszName = OBJ_nid2sn(nID)) == NULL)) {
      i2t_ASN1_OBJECT(chBuf, sizeof(chBuf), pEntry->object);
      pszName = chBuf;
    }
    strName = pszName;
    
    //TODO: implement more options. currently we use only the OID,
    // Shortname and long name shall also be the choices
      /*
    OBJ_obj2txt(chBuf, sizeof chBuf, pObject, 0);
    strName = chBuf;
      */

    int nType	= pEntry->value->type;
    int nLength = pEntry->value->length;
    unsigned char* pszData = pEntry->value->data;

    //setting up flags
    if ((nType == V_ASN1_GENERALSTRING) && ((nLength%4) == 0)){
      for(int nByteIdx = 0; nByteIdx < nLength; nByteIdx++){
	if(pszData[nByteIdx])
	  bDoIt[nByteIdx & 3] = true;
      }

      if(bDoIt[0] || bDoIt[1] || bDoIt[2]){
	bDoIt[0] = true;
	bDoIt[1] = true;
	bDoIt[2] = true;
	bDoIt[3] = true;
      } else {
	bDoIt[0] = false;
	bDoIt[1] = false;
	bDoIt[2] = false;
	bDoIt[3] = true;
      }
    } else {
      bDoIt[0] = true;
      bDoIt[1] = true;
      bDoIt[2] = true;
      bDoIt[3] = true;
    }

    strValue.erase();
    for(int nByteIdx = 0; nByteIdx < nLength; nByteIdx++) {
      if(!bDoIt[nByteIdx & 3])
	continue;
      unsigned char chTmp = pszData[nByteIdx];
      if(chTmp < ' ' || chTmp > '~'){
	char chTmpBuf[4];
	sprintf(chTmpBuf, "\\x%x", chTmp);
	strValue += (const char*)chTmpBuf;
      } else {
	strValue += chTmp;
      }
    }

    m_mapFields[strName].AddValue(strValue.c_str());
  }
  return true;
}

unsigned int 
CDavSSLDName::GetFieldCount()
{
  return m_mapFields.size();
}

void 
CDavSSLDName::Clear()
{
  m_mapFields.clear();
}

const CDavSSLDNameField* 
CDavSSLDName::GetField(const char* pszName)
{
  if(!pszName)
    return NULL;
  DNAMEFIELDS::iterator it = m_mapFields.find(pszName);
  if(it != m_mapFields.end())
    return &it->second;
  else
    return NULL;
}

const CDavSSLDNameField* 
CDavSSLDName::GetFirstField(OI_STRING_A& strName)
{
  m_itrCurrent = m_mapFields.begin();
  if(m_itrCurrent == m_mapFields.end()){
    return NULL;
  } else {
    strName = m_itrCurrent->first;
    return &m_itrCurrent->second;
  }
}

const CDavSSLDNameField* 
CDavSSLDName::GetNextField(OI_STRING_A& strName)
{
  if(m_itrCurrent == m_mapFields.end()){
    m_itrCurrent = m_mapFields.begin();
    return NULL;
  } else {
    m_itrCurrent++;
    if(m_itrCurrent == m_mapFields.end()){
	return NULL;
    } else {
      strName = m_itrCurrent->first;
      return &m_itrCurrent->second;
    }
  }
}

const CDavSSLDNameField* 
CDavSSLDName::operator [](const char* pszName)
{
  return GetField(pszName);
}

bool 
CDavSSLDName::operator ==(const CDavSSLDName& other)
{
  if(!m_pX509Name || !other.m_pX509Name)
    return false;
  else
    return (X509_NAME_cmp(m_pX509Name, other.m_pX509Name) == 0);
}

bool 
CDavSSLDName::operator ==(const X509_NAME* pOther)
{
  if(!m_pX509Name || !pOther)
    return false;
  else
    return (X509_NAME_cmp(m_pX509Name, pOther) == 0);
}

CDavSSLDName& 
CDavSSLDName::operator =(const CDavSSLDName& master)
{
  m_pX509Name = master.m_pX509Name;
  m_mapFields = master.m_mapFields;
  m_itrCurrent = m_mapFields.begin();
  return *this;
}


/* print subject name in oneline format */
bool
CDavSSLDName::OneLine(OI_STRING_A& strOut)
{
  if(!m_pX509Name)
    return false;

  char* pszName = X509_NAME_oneline(m_pX509Name, NULL, 0);
  if(!pszName)
    return false;
  strOut = pszName;
  OPENSSL_free(pszName);
  return true;
}
