/* 
   Configuration to handle different kinds of servers
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavServerPolicy.cpp 558 2017-10-29 01:11:56Z tfbuilder $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/DavServerPolicy.h>
#include <onion/DavXmlErrorHandler.h>
#include <onion/DavWorkSession.h>


CDavServerPolicy::CDavServerPolicy(void):
  m_xbfConfig(OI_XMLPOLICY_DOCNAME)
{
  m_pConfig = NULL;
  m_pRoot = NULL;
}

CDavServerPolicy::~CDavServerPolicy(void)
{
}

void 
CDavServerPolicy::Reset()
{
  m_pConfig = NULL;
  m_pRoot = NULL;
  m_pDefaultMethods = NULL;
  m_pDefaultStatuses = NULL;
  //m_xbfConfig.Clear();
}

bool 
CDavServerPolicy::Load(const char* pszPath)
{
  if(!m_xbfConfig.Load(pszPath))
    return false;
  m_pConfig = m_xbfConfig.GetDocument();
  if(initConfig())
    return true;

  Reset();
  return false;
}

bool
CDavServerPolicy::Load(CDavXmlBuffer* pXmlBuffer)
{
  if(!pXmlBuffer)
    return false;
  m_xbfConfig = *pXmlBuffer;
  m_pConfig = m_xbfConfig.GetDocument();
  if(initConfig())
    return true;

  Reset();
  return false;
}

void
CDavServerPolicy::Save(const char* pszPath)
{
  OI_ASSERT(pszPath);
  m_xbfConfig.SaveAs(pszPath);
}

/*
 * try to make sure all key element
 * pointers are in place
 */
bool 
CDavServerPolicy::secureConfig()
{
  if(m_pConfig)
    return true;

  generateDefault();

  if(initConfig())
    return true;

  Reset();
  return false;
}

/*
 * Initialize cached pointers to each
 * key element. Caller should reset
 * m_pConfig in case this function fails
 */
bool 
CDavServerPolicy::initConfig()
{
  OI_ASSERT(m_pConfig);
  m_pDefaultMethods = NULL;
  m_pDefaultStatuses = NULL;
  m_pRoot = m_pConfig->getDocumentElement();

  XNS(DOMNodeList) *pNodeList;
  XNS(DOMElement) *pDefaultServer;

  //getting the default server element
  pNodeList = m_pRoot->getElementsByTagName(X("DefaultServer"));
  if(pNodeList->getLength() < 1) {
    OI_DEBUG("default server configuration not found!\n");
    return false;
  }

  pDefaultServer = (XNS(DOMElement)*)(pNodeList->item(0));
  if(!pDefaultServer){
    OI_DEBUG("href Content not found!\n");
    return false;
  }

  //setting pointer for DefaultServer->Methods
  pNodeList = pDefaultServer->getElementsByTagName(X("Methods"));
  if(pNodeList->getLength() < 1){
    OI_DEBUG("default server configuration not found!\n");
    return false;
  }
  m_pDefaultMethods = (XNS(DOMElement)*)(pNodeList->item(0));
  if(!m_pDefaultMethods){
    OI_DEBUG("href Content not found!\n");
    return false;
  }

  //setting pointer for DefaultServer->Methods
  pNodeList = pDefaultServer->getElementsByTagName(X("CommonStatus"));
  if(pNodeList->getLength() < 1){
    OI_DEBUG("default server configuration not found!\n");
    return false;
  }
  m_pDefaultStatuses = (XNS(DOMElement)*)(pNodeList->item(0));
  if(!m_pDefaultStatuses){
    OI_DEBUG("href Content not found!\n");
    return false;
  }

  return true;
}

XNS(DOMElement)* 
CDavServerPolicy::appendElement(XNS(DOMElement)* pParent,
				const XMLCh* pszName)
{
  if((!pParent) || (!pszName))
    return NULL;
  XNS(DOMElement)* pChild = m_pConfig->createElement(pszName);
  pParent->appendChild(pChild);
  return pChild;
}

XNS(DOMElement)* 
CDavServerPolicy::appendCType(XNS(DOMElement)* pParent,
			      const XMLCh* pszType,
			      const XMLCh* pszErrCode,
			      const XMLCh* pszOpCode)
{
  if((!pParent)||(!pszType)||(!pszErrCode)||(!pszOpCode)) 
    return NULL;
  XNS(DOMElement)* pChild = m_pConfig->createElement(m_pszCType);
  pParent->appendChild(pChild);
  pChild->setAttribute(m_pszType, pszType);
  pChild->setAttribute(m_pszErrCode, pszErrCode);
  pChild->setAttribute(m_pszOpCode, pszOpCode);
  return pChild;
}

XNS(DOMElement)* 
CDavServerPolicy::appendMethod(XNS(DOMElement)* pParent,
			       const XMLCh* pszMethod,
			       const XMLCh* pszDefaultOp,
			       const XMLCh* pszDefaultErr)
{
  if((!pParent)||(!pszMethod)||(!pszDefaultOp)||(!pszDefaultErr)) 
    return NULL;
  XNS(DOMElement)* pChild = m_pConfig->createElement(pszMethod);
  pParent->appendChild(pChild);
  pChild->setAttribute(m_pszDefaultOp, pszDefaultOp);
  pChild->setAttribute(m_pszDefaultErr, pszDefaultErr);
  return pChild;
}

XNS(DOMElement)* 
CDavServerPolicy::appendStatus(XNS(DOMElement)* pParent,
			       const XMLCh* pszStatus,
			       const XMLCh* pszChkBody,
			       const XMLCh* pszDefErr,
			       const XMLCh* pszDefOp)
{
  if((!pParent)||(!pszStatus)||(!pszChkBody)||(!pszDefErr)||(!pszDefOp)) 
    return NULL;
  XNS(DOMElement)* pChild = m_pConfig->createElement(pszStatus);
  pParent->appendChild(pChild);
  pChild->setAttribute(m_pszChkBody, pszChkBody);
  pChild->setAttribute(m_pszDefErr, pszDefErr);
  pChild->setAttribute(m_pszDefOp, pszDefOp);
  return pChild;
}

/*
 * the hard-coded policy.
 * Fallback to this only when no
 * policy is loaded
 */
void 
CDavServerPolicy::generateDefault()
{
  OI_ASSERT(!m_pConfig);

  //initializing node names
  X strCType("CType");
  X strType("Type");
  X strErrCode("ErrCode");
  X strOpCode("OpCode");
  X strChkBody("ChkBody");
  X strDefErr("DefErr");
  X strDefOp("DefOp");
  X strDefaultErr("DefaultErr");
  X strDefaultOp("DefaultOp");

  m_pszCType = strCType;
  m_pszType = strType;
  m_pszErrCode = strErrCode;
  m_pszOpCode = strOpCode;
  m_pszChkBody = strChkBody;
  m_pszDefErr = strDefErr;
  m_pszDefOp = strDefOp;
  m_pszDefaultErr = strDefaultErr;
  m_pszDefaultOp = strDefaultOp;

  //initializing value strings
  X strT("T");
  X strF("F");
  X str0("0");
  X str1("1");
  X str2("2");
  X str3("3");
  X str4("4");
  X str5("5");
  X str6("6");
  X str7("7");
  X str901("901");
  X str902("902");
  X str903("903");
  X strTEXT_HTML("text/html");
  X strTEXT_XML("text/xml");

  const XMLCh* pszT = strT;
  const XMLCh* pszF = strF;
  const XMLCh* psz0 = str0;
  const XMLCh* psz1 = str1;
  const XMLCh* psz2 = str2;
  const XMLCh* psz3 = str3;
  const XMLCh* psz4 = str4;
  const XMLCh* psz5 = str5;
  const XMLCh* psz6 = str6;
  const XMLCh* psz7 = str7;
  const XMLCh* psz901 = str901;
  const XMLCh* psz902 = str902;
  const XMLCh* psz903 = str903;
  const XMLCh* pszTEXT_HTML = strTEXT_HTML;
  const XMLCh* pszTEXT_XML = strTEXT_XML;


  m_xbfConfig.Clear();
  m_pConfig = m_xbfConfig.GetDocument();
  OI_ASSERT(m_pConfig);

  //the DefaultServer Element
  XNS(DOMElement)* pRoot = m_pConfig->getDocumentElement();
  XNS(DOMElement)* pDefaultServer = appendElement(pRoot, X("DefaultServer"));
  pDefaultServer->setAttribute(X("Name"), X("Default Server"));

  //the CommonStatus Element
  XNS(DOMElement)* pCommonStatus = appendElement(pDefaultServer,
						 X("CommonStatus"));
  //S206 for CommonStatus
  XNS(DOMElement)* pS206 = appendStatus(pCommonStatus, 
					X("S206"),
					pszF,
					psz0,
					psz0);
  //S301 for CommonStatus
  XNS(DOMElement)* pS301 = appendStatus(pCommonStatus, 
					X("S301"),
					pszT,
					psz903,
					psz1);
  //CType for S301
  XNS(DOMElement)* pS301CType = appendCType(pS301, 
					    pszTEXT_HTML,
					    X("819"),
					    psz4);
  //S302 for CommonStatus
  XNS(DOMElement)* pS302 = appendStatus(pCommonStatus, 
					X("S302"),
					pszT,
					psz903,
					psz1);
  //CType for S302
  XNS(DOMElement)* pS302CType = appendCType(pS302, 
					    pszTEXT_HTML,
					    X("820"),
					    psz4);
  //S400 for CommonStatus
  XNS(DOMElement)* pS400 = appendStatus(pCommonStatus, 
					X("S400"),
					pszT,
					psz903,
					psz1);
  //CType for S400
  XNS(DOMElement)* pS400CType = appendCType(pS400, 
					    pszTEXT_HTML,
					    X("803"),
					    psz4);
  //S401 for CommonStatus
  XNS(DOMElement)* pS401 = appendStatus(pCommonStatus, 
					X("S401"),
					pszT,
					psz903,
					psz1);
  //CType for S401
  XNS(DOMElement)* pS401CType = appendCType(pS401, 
					    pszTEXT_HTML,
					    psz0,
					    psz7);
  //S402 for CommonStatus
  XNS(DOMElement)* pS402 = appendStatus(pCommonStatus, 
					X("S402"),
					pszT,
					psz903,
					psz1);
  //CType for S402
  XNS(DOMElement)* pS402CType = appendCType(pS402, 
					    pszTEXT_HTML,
					    X("822"),
					    psz4);
  //S403 for CommonStatus
  XNS(DOMElement)* pS403 = appendStatus(pCommonStatus, 
					X("S403"),
					pszT,
					psz903,
					psz1);
  //CType for S403
  XNS(DOMElement)* pS403CType = appendCType(pS403, 
					    pszTEXT_HTML,
					    X("804"),
					    psz4);
  //S404 for CommonStatus
  XNS(DOMElement)* pS404 = appendStatus(pCommonStatus, 
					X("S404"),
					pszT,
					psz903,
					psz1);
  //CType for S404
  XNS(DOMElement)* pS404CType = appendCType(pS404, 
					    pszTEXT_HTML,
					    X("805"),
					    psz4);
  //S405 for CommonStatus
  XNS(DOMElement)* pS405 = appendStatus(pCommonStatus, 
					X("S405"),
					pszT,
					psz903,
					psz1);
  //CType for S405
  XNS(DOMElement)* pS405CType = appendCType(pS405, 
					    pszTEXT_HTML,
					    X("806"),
					    psz4);
  //S407 for CommonStatus
  XNS(DOMElement)* pS407 = appendStatus(pCommonStatus, 
					X("S407"),
					pszT,
					psz903,
					psz1);
  //CType for S407
  XNS(DOMElement)* pS407CType = appendCType(pS407, 
					    pszTEXT_HTML,
					    psz0,
					    psz7);
  //S409 for CommonStatus
  XNS(DOMElement)* pS409 = appendStatus(pCommonStatus, 
					X("S409"),
					pszT,
					psz903,
					psz1);
  //CType for S409
  XNS(DOMElement)* pS409CType = appendCType(pS409, 
					    pszTEXT_HTML,
					    X("807"),
					    psz4);
  //S500 for CommonStatus
  XNS(DOMElement)* pS500 = appendStatus(pCommonStatus, 
					X("S500"),
					pszT,
					psz903,
					psz1);
  //CType for S500
  XNS(DOMElement)* pS500CType = appendCType(pS500, 
					    pszTEXT_HTML,
					    X("812"),
					    psz4);
  //S502 for CommonStatus
  XNS(DOMElement)* pS502 = appendStatus(pCommonStatus, 
					X("S502"),
					pszT,
					psz903,
					psz1);
  //CType for S502
  XNS(DOMElement)* pS502CType = appendCType(pS502, 
					    pszTEXT_HTML,
					    X("813"),
					    psz4);
  //S503 for CommonStatus
  XNS(DOMElement)* pS503 = appendStatus(pCommonStatus, 
					X("S503"),
					pszT,
					psz903,
					psz1);
  //CType for S503
  XNS(DOMElement)* pS503CType = appendCType(pS503, 
					    pszTEXT_HTML,
					    X("814"),
					    psz4);
  //S507 for CommonStatus
  XNS(DOMElement)* pS507 = appendStatus(pCommonStatus, 
					X("S507"),
					pszT,
					psz903,
					psz1);
  //CType for S507
  XNS(DOMElement)* pS507CType = appendCType(pS507, 
					    pszTEXT_HTML,
					    X("821"),
					    psz4);
  // S421 for CommonStatus
  XNS(DOMElement)* psS421 = appendStatus(pCommonStatus,
					X("S421"),
					pszT,
					psz903,
					psz1);
  //CType for S421
  XNS(DOMElement)* lpS421CType = appendCType(psS421,
						pszTEXT_HTML,
						X("823"),
						psz4);

  //the Methods Element
  XNS(DOMElement)* pMethods = appendElement(pDefaultServer, 
					    X("Methods"));
  pMethods->setAttribute(m_pszDefaultOp, psz1);
  pMethods->setAttribute(m_pszDefaultErr, psz901);

  /////////////////////////////////////////////////////////////////
  //the CONNECT method
  XNS(DOMElement)* pCONNECT = appendMethod(pMethods, 
					   X("CONNECT"),
					   psz1,
					   psz902);
  //S200 for CONNECT
  XNS(DOMElement)* pS200 = appendStatus(pCONNECT, 
					X("S200"),
					pszF,
					psz0,
					psz0);

  /////////////////////////////////////////////////////////////////
  //the COPY method
  XNS(DOMElement)* pCOPY = appendMethod(pMethods, 
					X("COPY"),
					psz1,
					psz902);
  //S201 for COPY
  XNS(DOMElement)* pS201 = appendStatus(pCOPY, 
					X("S201"),
					pszT,
					psz903,
					psz1);
  //CType for S201
  XNS(DOMElement)* pS201CType = appendCType(pS201, 
					    pszTEXT_HTML,
					    psz0,
					    psz4);
  //S204 for COPY
  XNS(DOMElement)* pS204 = appendStatus(pCOPY, 
					X("S204"),
					pszF,
					psz0,
					psz0);
  //S412 for COPY
  XNS(DOMElement)* pS412 = appendStatus(pCOPY, 
					X("S412"),
					pszT,
					psz903,
					psz1);
  //CType for S412
  XNS(DOMElement)* pS412CType = appendCType(pS412, 
					    pszTEXT_HTML,
					    X("808"),
					    psz4);

  //S424 for COPY
  XNS(DOMElement)* pS424 = appendStatus(pCOPY, 
					X("S424"),
					pszT,
					psz903,
					psz1);
  //CType for S424
  XNS(DOMElement)* pS424CType = appendCType(pS424, 
					    pszTEXT_XML,
					    X("810"),
					    psz3);
  pS424CType->setAttribute(X("CutElement"), X("response"));

  /////////////////////////////////////////////////////////////////
  //the DELETE method
  XNS(DOMElement)* pDELETE = appendMethod(pMethods, 
					  X("DELETE"),
					  psz1,
					  psz902);
  //S204 for DELETE
  pDELETE->appendChild(pS204->cloneNode(true));
  //S423 for DELETE
  XNS(DOMElement)* pS423 = appendStatus(pDELETE, 
					X("S423"),
					pszT,
					psz903,
					psz1);
  //CType for S423
  XNS(DOMElement)* pS423CType = appendCType(pS423, 
					    pszTEXT_HTML,
					    X("809"),
					    psz4);

  //S424 for DELETE
  pDELETE->appendChild(pS424->cloneNode(true));

  /////////////////////////////////////////////////////////////////
  //the GET method
  XNS(DOMElement)* pGET = appendMethod(pMethods, 
				       X("GET"),
				       psz1,
				       psz902);
  //S200 for GET
  pGET->appendChild(pS200->cloneNode(true));


  /////////////////////////////////////////////////////////////////
  //the HEAD method
  XNS(DOMElement)* pHEAD = appendMethod(pMethods, 
					X("HEAD"),
					psz1,
					psz902);
  //S200 for HEAD
  pHEAD->appendChild(pS200->cloneNode(true));

  /////////////////////////////////////////////////////////////////
  //the LOCK method
  XNS(DOMElement)* pLOCK = appendMethod(pMethods, 
					X("LOCK"),
					psz1,
					psz902);
  //S200 for LOCK
  XNS(DOMElement)* pS200Lock = appendStatus(pLOCK, 
					    X("S200"),
					    pszT,
					    psz903,
					    psz1);
  //CType for S200
  XNS(DOMElement)* pS200LockCType = appendCType(pS200Lock, 
						pszTEXT_XML,
						psz0,
						psz0);
  //S207 for LOCK
  XNS(DOMElement)* pS207Lock = appendStatus(pLOCK, 
					    X("S207"),
					    pszT,
					    psz903,
					    psz1);
  //CType for S207
  XNS(DOMElement)* pS207LockCType = appendCType(pS207Lock, 
						pszTEXT_XML,
						X("802"),
						psz3);
  pS207LockCType->setAttribute(X("CutElement"), X("response"));
  //lpS207LockCType->setAttribute(L"CutElement", L"multistatus");
  //S423 for LOCK
  pLOCK->appendChild(pS423->cloneNode(true));

  /////////////////////////////////////////////////////////////////
  //the MKCOL method
  XNS(DOMElement)* pMKCOL = appendMethod(pMethods, 
					 X("MKCOL"),
					 psz1,
					 psz902);
  //S201 for MKCOL
  pMKCOL->appendChild(pS201->cloneNode(true));
  //S207 for MKCOL(copy from LOCK method)
  pMKCOL->appendChild(pS207Lock->cloneNode(true));

  /////////////////////////////////////////////////////////////////
  //the MOVE method
  XNS(DOMElement)* pMOVE = appendMethod(pMethods, 
					X("MOVE"),
					psz1,
					psz902);
  //S201 for MOVE
  pMOVE->appendChild(pS201->cloneNode(true));
  //S204 for MOVE
  pMOVE->appendChild(pS204->cloneNode(true));
  //S412 for MOVE
  pMOVE->appendChild(pS412->cloneNode(true));
  //S423 for MOVE
  pMOVE->appendChild(pS423->cloneNode(true));
  //S424 for MOVE
  pMOVE->appendChild(pS424->cloneNode(true));

  /////////////////////////////////////////////////////////////////
  //the OPTIONS method
  XNS(DOMElement)* pOPTIONS = appendMethod(pMethods, 
					   X("OPTIONS"),
					   psz1,
					   psz902);
  //S200 for OPTIONS
  pOPTIONS->appendChild(pS200->cloneNode(true));

  /////////////////////////////////////////////////////////////////
  //the PROPFIND method
  XNS(DOMElement)* pPROPFIND = appendMethod(pMethods, 
					    X("PROPFIND"),
					    psz1,
					    psz902);
  //S207 for PROPFIND
  XNS(DOMElement)* pS207PFind = appendStatus(pPROPFIND, 
					     X("S207"),
					     pszT,
					     psz903,
					     psz1);
  //CType for S200
  XNS(DOMElement)* pS207PFindCType = appendCType(pS207PFind, 
						 pszTEXT_XML,
						 psz0,
						 psz0);
  pS207PFindCType->setAttribute(X("CutElement"), X("response"));


  /////////////////////////////////////////////////////////////////
  //the PROPPATCH method
  XNS(DOMElement)* pPROPPATCH = appendMethod(pMethods, 
					     X("PROPPATCH"),
					     psz1,
					     psz902);
  //S207 for PROPPATCH
  pPROPPATCH->appendChild(pS207PFind->cloneNode(true));
  //S423 for PROPPATCH
  pPROPPATCH->appendChild(pS423->cloneNode(true));

  /////////////////////////////////////////////////////////////////
  //the PUT method
  XNS(DOMElement)* pPUT = appendMethod(pMethods, 
				       X("PUT"),
				       psz1,
				       psz902);
  //S201 for PUT
  pPUT->appendChild(pS201->cloneNode(true));
  //S204 for PUT
  XNS(DOMElement)* pS204Put = appendStatus(pPUT, 
					   X("S204"),
					   pszF,
					   X("801"),
					   psz0);
  //S207 for PUT(copy from LOCK method)
  pPUT->appendChild(pS207Lock->cloneNode(true));
  //S423 for PUT(copy from LOCK method)
  pPUT->appendChild(pS423->cloneNode(true));



  /////////////////////////////////////////////////////////////////
  //the SEARCH method
  XNS(DOMElement)* pSEARCH = appendMethod(pMethods, 
					  X("SEARCH"),
					  psz1,
					  psz902);
  //S207 for SEARCH
  pSEARCH->appendChild(pS207PFind->cloneNode(true));


  /////////////////////////////////////////////////////////////////
  //the UNLOCK method
  XNS(DOMElement)* pUNLOCK = appendMethod(pMethods, 
					  X("UNLOCK"),
					  psz1,
					  psz902);
  //S204 for UNLOCK
  pUNLOCK->appendChild(pS204Put->cloneNode(true));

}

void 
CDavServerPolicy::WriteDefaultConfig(const char* pszPath)
{
  /*
  generateDefault();
  XNS(DOMDocument)* pDoc = m_pConfig;
  if(!initConfig(pDoc))
    return;

  XNS(DOMWriter)* pWriter;
  XNS(LocalFileFormatTarget)* pFormatTarget;
  XNS(DOMImplementation) *pImpl = 
    XNS(DOMImplementationRegistry)::getDOMImplementation(X("LS"));
  pWriter = ((XNS(DOMImplementationLS)*)pImpl)->createDOMWriter();
  pFormatTarget = new XNS(LocalFileFormatTarget)(pszPath);
  pWriter->setNewLine(X("\r\n"));
  pWriter->setFeature(XNS(XMLUni)::fgDOMWRTFormatPrettyPrint, true);
  XNS(DOMDocument)* pDoc2 = pDoc->getOwnerDocument();
  pWriter->writeNode(pFormatTarget, *pDoc);

  pWriter->release();
  delete pFormatTarget;
  */
}

OI_RESULT 
CDavServerPolicy::CheckResponse(CDavWorkSession* pSession,
				const char* pszServerString,
				const char* pszRequest,
				int nStatus,
				const char* pszContentType,
				OIOPERATION *pOperation)
{
  int i;
  OI_ASSERT(pOperation);
  OI_ASSERT(pSession);
  OI_STRING_A strResponseCType = pszContentType;

  if(!secureConfig())
    return OIYECONFINITFAILED;

  MakeLowerA(strResponseCType);
  pOperation->szCutElement[0] = 0;
  XNS(DOMElement)* pCommonStatus = NULL;

  //lookup server
  bool bServerFound = false;
  OI_STRING_A strServer = pszServerString;
  XNS(DOMNodeList) *pNodeList;
  XNS(DOMElement) *pServer = NULL;
  XNS(DOMElement) *pMethods = NULL;

  pNodeList = m_pRoot->getElementsByTagName(X("Server"));

  int nServerCount = pNodeList->getLength();
  for(i = 0; i < nServerCount; i++){
    pServer = (XNS(DOMElement)*)(pNodeList->item(i));
    OI_STRING_A strName = (const char*)X(pServer->getAttribute(X("Name")));
    if(pSession->MatchServerString(strServer.c_str(), strName.c_str())){
      OI_DEBUG("Server Config [%s] found\n", strName.c_str());
      bServerFound = true;
      break;
    }
  }
  if(bServerFound){
    pNodeList = pServer->getElementsByTagName(X("Methods"));
    pMethods = (XNS(DOMElement)*)(pNodeList->item(0));
    if(!pMethods)
      return OIYECONFSYNTAX;
    pNodeList = pServer->getElementsByTagName(X("CommonStatus"));
    pCommonStatus = (XNS(DOMElement)*)(pNodeList->item(0));
    if(!pCommonStatus)
      pCommonStatus = m_pDefaultStatuses;
  } else {
    OI_DEBUG("Server Config not found, use default\n");
    pMethods		= m_pDefaultMethods;
    pCommonStatus	= m_pDefaultStatuses;
  }
	
  OI_STRING_A strDefaultOp =
    (const char*)X(pMethods->getAttribute(X("DefaultOp")));
  OI_STRING_A strDefaultErr =
    (const char*)X(pMethods->getAttribute(X("DefaultErr")));

  if(strDefaultOp.empty() || strDefaultErr.empty()){
    return OIYECONFSYNTAX;
  }

  long lDefOp = atol(strDefaultOp.c_str());
  long lDefEr = atol(strDefaultErr.c_str());

  m_enuOpCode = (OI_OPCODE)lDefOp;
  m_enuError = (OI_RESULT)lDefEr;

  //lookup method
  OI_STRING_A strMethod = pszRequest;
  pNodeList = pMethods->getElementsByTagName(X(strMethod.c_str()));

  XNS(DOMElement)* pMethod = (XNS(DOMElement)*)(pNodeList->item(0));
  if(!pMethod){
    pOperation->enuOpCode = m_enuOpCode;
    pOperation->enuErrCode = m_enuError;
    return OI_OK;
  }
	
  strDefaultOp = X(pMethod->getAttribute(X("DefaultOp")));
  strDefaultErr = X(pMethod->getAttribute(X("DefaultErr")));

  if(strDefaultOp.empty() || strDefaultErr.empty()){
    return OIYECONFSYNTAX;
  }

  lDefOp = atol(strDefaultOp.c_str());
  lDefEr = atol(strDefaultErr.c_str());

  m_enuOpCode = (OI_OPCODE)lDefOp;
  m_enuError = (OI_RESULT)lDefEr;


  //lookup status
  //OI_STRING_A strStatus;
  char chStatus[5];
  sprintf(chStatus, "S%03u", nStatus);
  /*
    OI_STRSTREAM_A ss;
    ss << "S" << setfill('0') << setw(3) << nStatus;
    strStatus = ss.str();
  */

  //pNodeList = pCommonStatus->getElementsByTagName(X(strStatus.c_str()));
  pNodeList = pCommonStatus->getElementsByTagName(X(chStatus));
  XNS(DOMElement)* pStatus = (XNS(DOMElement)*)(pNodeList->item(0));

  if(!pStatus){
    //pNodeList = pMethod->getElementsByTagName(X(strStatus.c_str()));
    pNodeList = pMethod->getElementsByTagName(X(chStatus));
    pStatus = (XNS(DOMElement)*)(pNodeList->item(0));
  }
  if(!pStatus){
    pOperation->enuOpCode	= m_enuOpCode;
    pOperation->enuErrCode	= m_enuError;
    return OI_OK;
  }

  strDefaultOp	= X(pStatus->getAttribute(X("DefOp")));
  strDefaultErr	= X(pStatus->getAttribute(X("DefErr")));
  
  if(strDefaultOp.empty() || strDefaultErr.empty()){
    return OIYECONFSYNTAX;
  }

  lDefOp = atol(strDefaultOp.c_str());
  lDefEr = atol(strDefaultErr.c_str());

  m_enuOpCode = (OI_OPCODE)lDefOp;
  m_enuError = (OI_RESULT)lDefEr;

  bool bChkBody = false;
  OI_STRING_A strChkBody = (const char*)X(pStatus->getAttribute(X("ChkBody")));
  MakeLowerA(strChkBody);

  if(!strChkBody.empty()){
    if(strChkBody == "t")
      bChkBody = true;
  }

  //check content type
  if(!bChkBody){
    pOperation->enuOpCode	= m_enuOpCode;
    pOperation->enuErrCode	= m_enuError;
    return OI_OK;
  }

  pNodeList = pStatus->getElementsByTagName(X("CType"));
  int nLen = pNodeList->getLength();

  for(i = 0; i < nLen; i++){
    XNS(DOMElement)* pCType = (XNS(DOMElement)*)(pNodeList->item(i));
    OI_STRING_A strCType = (const char*)X(pCType->getAttribute(X("Type")));
    if(strCType.empty())
      continue;

    MakeLowerA(strCType);
    if(strCType == strResponseCType){
      OI_STRING_A strOp	= (const char*)X(pCType->getAttribute(X("OpCode")));
      OI_STRING_A strErr = (const char*)X(pCType->getAttribute(X("ErrCode")));
      if(strOp.empty() || strErr.empty()){
	return OIYECONFSYNTAX;
      }
      lDefOp = atol(strOp.c_str());
      lDefEr = atol(strErr.c_str());
      OI_STRING_A strCutElement =
	(const char*)X(pCType->getAttribute(X("CutElement")));
      if(strCutElement.length() > OI_MAX_ELEM_LEN){
	strCutElement.erase(OI_MAX_ELEM_LEN);
      }
      pOperation->enuOpCode = (OI_OPCODE)lDefOp;
      pOperation->enuErrCode = (OI_RESULT)lDefEr;
      strncpy(pOperation->szCutElement,
	      strCutElement.c_str(),
	      OI_MAX_ELEM_LEN);
      return OI_OK;
    }
  }
  //no match, return the default
  pOperation->enuOpCode	= m_enuOpCode;
  pOperation->enuErrCode = m_enuError;
  return OI_OK;
}

