/* 
   Socket wrapper for http operations
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 1998-2004, Joe Orton <joe@manyfish.co.uk>,
   Copyright (C) 1999-2000, Tommi Komulainen <Tommi.Komulainen@iki.fi>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavSocket.h 132 2005-06-24 09:09:43Z komat $
*/

#ifndef ONION_HEADER_DAVSOCKET_H
#define ONION_HEADER_DAVSOCKET_H 1

#include <onion/OnionCfg.h>
#include <openssl/ssl.h>

#if defined(__BEOS__) && !defined(BONE_VERSION)
#  define ONION_WRITE(a,b,c) send(a,b,c,0)
#  define ONION_READ(a,b,c) recv(a,b,c,0)
#  define ONION_CLOSE(s) closesocket(s)
#  define ONION_INVALIDSOCKET -1
#  define onion_errno errno
#  define ONION_SOCKERR (-1)
#  define ONION_EINTR EINTR
#  define ONION_ISRESET(e) ((e) == ECONNRESET)
#elif defined(WIN32)
#  define ONION_WRITE(a,b,c) send(a,b,c,0)
#  define ONION_READ(a,b,c) recv(a,b,c,0)
#  define ONION_CLOSE(s) closesocket(s)
#  define ONION_INVALIDSOCKET INVALID_SOCKET
#  define onion_errno WSAGetLastError()
#  define ONION_SOCKERR SOCKET_ERROR
#  define ONION_EINTR WSAEINTR
#  define ONION_ISRESET(e) ((e == WSAECONNABORTED) || (e == WSAETIMEDOUT) ||\
			    (e == WSAECONNRESET) || (e == WSAENETRESET))
#elif defined(__OTCMSOCKET__)
#  define ONION_WRITE(a,b,c) send(a,b,c, 0)
#  define ONION_READ(a,b,c) recv(a,b,c, 0)
#  define ONION_CLOSE(s) closesocket(s)
#  define ONION_INVALIDSOCKET -1
#  define onion_errno getsockerrno()
#  define ONION_SOCKERR (-1)
#  define ONION_EINTR EINTR
#  define ONION_ISRESET(e) ((e) == ECONNRESET)
#else  /* UNIX */
#  define ONION_WRITE(a,b,c) send(a,b,c, 0)
#  define ONION_READ(a,b,c) recv(a,b,c, 0)
#  define ONION_CLOSE(s) close(s)
#  define ONION_INVALIDSOCKET -1
#  define onion_errno errno
#  define ONION_SOCKERR (-1)
#  define ONION_EINTR EINTR
#  define ONION_ISRESET(e) ((e) == ECONNRESET)
#endif

//forward declarations
class IDavSocketIO;
class CDavSocketIOSSL;
class CDavSocketIORaw;
class CDavWorkSession;

/** Internal wrapper of socket
 */
OI_BEGIN_CLASSEXPORT

class OI_EXPORT CDavSocket
{
  friend class IDavSocketIO;
  friend class CDavSocketIOSSL;
  friend class CDavSocketIORaw;
  friend class CDavWorkSession;
 public:
  CDavSocket(CDavWorkSession* pSession);
  virtual ~CDavSocket(void);
  void ResetBuffer();
  void Disconnect();
  bool UseSSL(bool bUse);
  bool IsUsingSSL();
  bool HasSSLConnection();
  OI_RESULT Connect(const char* pszAddr, unsigned short usPort);
  int GetHandle();
  unsigned int GetTimeout();
  void SetTimeout(unsigned int unTimeout);
  size_t GetReadBytes();
  size_t GetWrittenBytes();
  CDavWorkSession* GetSession();
  OI_RESULT Read(char *pszBuf, size_t *punReadLen);
  OI_RESULT ReadLine(char *pszBuf, size_t *punReadLen);
  OI_RESULT FullRead(char *pszBuf, size_t *punReadLen);
  OI_RESULT Peek(char *pszBuf, size_t *punReadLen);
  OI_RESULT Write(const char *pszBuf, size_t *punWrittenLen);
  void AddUnauthenticCert(X509* pX509, unsigned long ulFailures);
 protected:
  OI_RESULT connect(const struct sockaddr* pSockaddr, int nSockaddrLen);
  OI_RESULT connectSSL();
  SSL* getSSL();
  void cleanupSSL();
  void cleanupSSLSession();
 private:
  OI_RESULT initSSL();
 private:
  CDavWorkSession* m_pSession;
  CDavSocketIORaw* m_pIOFuncRaw;
  CDavSocketIOSSL* m_pIOFuncSSL;
  IDavSocketIO* m_pIOFunc;
  int m_hSocket;
  bool m_bConnected;
  char* m_pszBuffer;
  char* m_pszBufPos;
  size_t m_unContentLen;
  size_t m_unWriteCounter;
  size_t m_unReadCounter;
  unsigned int m_unTimeout;
  SSL* m_pSSL;
  SSL_SESSION* m_pSSLSession;
  X509* m_pServerCertificate;
};

OI_END_CLASSEXPORT

#endif /*ONION_HEADER_DAVSOCKET_H*/
