/* 
   SSL socket I/O methods
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 1998-2004, Joe Orton <joe@manyfish.co.uk>,
   Copyright (C) 1999-2000, Tommi Komulainen <Tommi.Komulainen@iki.fi>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavSocketIOSSL.cpp 132 2005-06-24 09:09:43Z komat $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif

#include <onion/DavSocketIOSSL.h>
#include <onion/DavSocket.h>
#include <onion/DavWorkSession.h>
#include <openssl/err.h>

CDavSocketIOSSL::CDavSocketIOSSL(void)
{
}

CDavSocketIOSSL::~CDavSocketIOSSL(void)
{
}

OI_RESULT 
CDavSocketIOSSL::ossl_error(CDavSocket *pSocket, int nErrCode)
{
  int nErr = SSL_get_error(pSocket->m_pSSL, nErrCode);
  OI_RESULT enuRet = OISEGEN;
	
  switch (nErr){
  case SSL_ERROR_ZERO_RETURN:
    enuRet = OISECLOSED;
    break;
  case SSL_ERROR_SYSCALL:
    nErr = ERR_get_error();
    if (nErr == 0){
      if (nErrCode == 0){
	/* EOF without close_notify, possible truncation */
	enuRet = OISETRUNC;
      } else {
	/* Other socket error. */
	//nErr = WSAGetLastError();
	enuRet = OISEGEN;
      }
    } else {
      enuRet = OIEEGENERIC;
    }
    break;
  default:
    enuRet = OIEEGENERIC;
    break;
  }
  return enuRet;
}

OI_RESULT 
CDavSocketIOSSL::Readable(CDavSocket* pSocket, int nSecs)
{
  if(pSocket->m_pSSL){
    if(SSL_pending(pSocket->m_pSSL)){
      return OI_OK;
    }
  }
  
  /*
   * The following part is same as that of readable_raw()
   */
  int nRet;
  fd_set rdfds;
  int s = pSocket->GetHandle();
  struct timeval timeout, *tvp = (nSecs >= 0 ? &timeout : 0);
  
  /* Init the fd set */
  FD_ZERO(&rdfds);
  do{
    FD_SET(s, &rdfds);
    if (tvp){
      tvp->tv_sec = nSecs;
      tvp->tv_usec = 0;
    }
    //the first parameter of "select" is IGNORED on Windows platforms
    //it was merely here for compatibility with the past
    nRet = ::select(s + 1, &rdfds, 0, 0, tvp);
  } while (nRet < 0 && onion_errno == ONION_EINTR);
  // MSDN said return value would be SOCKET_ERROR if select failed.
  // Anyway, it is a negative value
  if (nRet < 0){
    //TODO: detailed error information?
    return OISEREADFAILED;
  }
  if(nRet == 0){
    return OISETIMEOUT;
  } else {
    return OI_OK;
  }
}

OI_RESULT 
CDavSocketIOSSL::Read(CDavSocket* pSocket,
		      char* pszBuf,
		      size_t* punReadLen,
		      int nSecs)
{
  OI_RESULT enuRet;
  int nBytes;
  int unBufLen = (int)(*punReadLen);
  int s = pSocket->GetHandle();
  enuRet = Readable(pSocket, nSecs);
  if(OI_OK != enuRet)
    return enuRet;


  nBytes = SSL_read(pSocket->m_pSSL, pszBuf, unBufLen);
  if(nBytes <= 0){
    enuRet = ossl_error(pSocket, nBytes);
    pSocket->m_pSession->Disconnect();
  } else {
    enuRet = OI_OK;
    *punReadLen = nBytes;
  }
  return enuRet;
}

OI_RESULT 
CDavSocketIOSSL::Write(CDavSocket* pSocket,
		       const char* pszBuf,
		       size_t* punWrittenLen)
{
  OI_ASSERT(pszBuf && punWrittenLen);
  int nBufLen  = (int)(*punWrittenLen);
  int nBytes = SSL_write(pSocket->m_pSSL, pszBuf, nBufLen);
  /* ssl.h says SSL_MODE_ENABLE_PARTIAL_WRITE must be enabled to
   * have SSL_write return < length...  so, SSL_write should never
   * return < length. */
  if (nBytes != nBufLen){
    pSocket->m_pSession->Disconnect();
    return ossl_error(pSocket, nBytes);
  } else {
    *punWrittenLen = nBytes;
    return OI_OK;
  }
}

