/* 
   Dav session handler
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 1999-2003, Joe Orton <joe@manyfish.co.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavWorkSession.cpp 460 2010-02-03 05:55:31Z yone $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/DavWorkSession.h>
#include <onion/DavHttpStream.h>
#include <onion/DavRequest.h>
#include <onion/DavSSLCertificate.h>
#include <onion/DavSSLCertificateList.h>
#include <onion/DavSSLClientCertificate.h>
#include <onion/RBPVDDummy.h>
#include <onion/RBPVDLocalFile.h>
#include <onion/RBPVDXmlCopy.h>
#include <onion/RBPVDXmlLock.h>
#include <onion/RBPVDXmlMove.h>
#include <onion/RBPVDXmlPropfind.h>
#include <onion/RBPVDXmlProppatch.h>
#include <onion/RBPVDXmlSearch.h>
#include <onion/PBCSMDummy.h>
#include <onion/PBCSMLocalFile.h>
#include <onion/PBCSMXml.h>
#include <onion/DavResourceNode.h>
#include <onion/DavSocket.h>
#include <onion/DavServerPolicy.h>
#include <onion/DavHost.h>
#include <onion/DavAuthManager.h>
#include <onion/DavXmlBuffer.h>
#include <onion/DavMemBuffer.h>
#include <onion/HandlerCRange.h>
#include <fcntl.h>

#ifdef WIN32
#include <io.h>
#elif defined(HAVE_UNISTD_H)
#include <unistd.h>
#endif /*HAVE_UNISTD_H*/

CDavWorkSession::CDavWorkSession(void)
{
  init();
}

CDavWorkSession::~CDavWorkSession(void)
{
  delete m_pSocket;
  delete m_pAuthManager;
  delete m_pXmlBuffer;
  delete m_pMemBuffer;
  delete m_pServerPolicy;
  delete m_pUnauthenticCerts;
}

bool
CDavWorkSession::IsExpecting100()
{
  return m_bExpect100;
}

bool
CDavWorkSession::IsPersistent()
{
  return m_bPersistent;
}

bool
CDavWorkSession::IsHttp11()
{
  return m_bHttp11;
}

#ifdef HAVE_LIBZ

bool
CDavWorkSession::IsUsingCompression()
{
  return m_bUseCompression;
}

void
CDavWorkSession::SetUseCompression(bool bValue)
{
  m_bUseCompression = bValue;
}

#endif /*HAVE_LIBZ*/

bool
CDavWorkSession::IsUsingSSL()
{
  return m_bUseSSL;
}

bool
CDavWorkSession::IsUsingProxy()
{
  return m_bUseProxy;
}

void
CDavWorkSession::SetExpecting100(bool bValue)
{
  m_bExpect100 = bValue;
}

void
CDavWorkSession::SetPersistent(bool bValue)
{
  m_bPersistent = bValue;
}

void
CDavWorkSession::SetHttp11(bool bValue)
{
  m_bHttp11 = bValue;
}

void
CDavWorkSession::SetUseSSL(bool bValue)
{
  m_bUseSSL = bValue;
}

void
CDavWorkSession::SetUseProxy(bool bValue)
{
  m_bUseProxy = bValue;
}

void
CDavWorkSession::SetSocketTimeout(unsigned int unTimeout)
{
  m_pSocket->SetTimeout(unTimeout);
}

void 
CDavWorkSession::Disconnect()
{
  if(m_enuConnState != S_CONN_DISCONNECTED){
    m_pSocket->Disconnect();
    m_enuConnState = S_CONN_DISCONNECTED;
  }
  m_tmConnectionTTL = -1;
  m_nMaxConnections = -1;
  OI_DEBUG("Disconnected\n");
}

OI_CONN_STATE
CDavWorkSession::GetConnectState()
{
  return m_enuConnState;
}

bool 
CDavWorkSession::OnAuthentication(X& strRealm,
				  OI_STRING_A& strUsername,
				  OI_STRING_A& strPasswd,
				  int nRetry,
				  OI_AUTH_CLASS enuAuthClass)
{
  return true;
}

void 
CDavWorkSession::OnCancelSend(CDavRequest* pReq,
			      CRequestBodyProvider* pRBProvider)
{
}

void 
CDavWorkSession::OnCancelReceive(CDavRequest* pReq,
				 CResponseBodyConsumer* pPBConsumer)
{
}

void
CDavWorkSession::OnCreateRequest(CDavRequest* pReq)
{
}

void
CDavWorkSession::OnPreBuildRequest(CDavRequest* pReq)
{
}

void
CDavWorkSession::OnPreSendRequest(CDavRequest* pReq)
{
}

void
CDavWorkSession::OnPostSendRequest(CDavRequest* pReq)
{
}

void
CDavWorkSession::OnPreRecvResponse(CDavRequest* pReq)
{
}

void
CDavWorkSession::OnPostRecvResponse(CDavRequest* pReq)
{
}

void
CDavWorkSession::OnDestroyRequest(CDavRequest* pReq)
{
}

OI_RESULT
CDavWorkSession::ProcessExtraHeader(CDavRequest* pReq,
				    const char* pszName,
				    OI_STRLIST_A& vecValues)
{
  return OI_OK;
}

bool
CDavWorkSession::MatchServerString(const char* pszHeaderValue,
				   const char* pszConfigValue)
{
  return (strcmp(pszHeaderValue, pszConfigValue) == 0);
}


bool
CDavWorkSession::OnSendProgress(CDavRequest* pReq,
				OI_SIZE_T unProgress,
				OI_SIZE_T unTotal)
{
  return true;
}

bool
CDavWorkSession::OnReceiveProgress(CDavRequest* pReq,
				   OI_SIZE_T unProgress,
				   OI_SIZE_T unTotal,
				   bool bCLength)
{
  return true;
}

void
CDavWorkSession::OnResourceFound(CDavRequest* pReq, CDavResourceNode* pNode)
{
}

void
CDavWorkSession::OnSearchResponse(CDavRequest* pReq, XNS(DOMNode)* pNode)
{
}

//SSL callbacks
bool
CDavWorkSession::OnVerifyCertificates(CDavSSLCertificateList* pUnauthenticCerts)
{
  return false;
}

bool
CDavWorkSession::OnProvideClientCertificate(CDavSSLClientCertificate* pCertOut,
					    CDavSSLDName* pCAList,
					    int nCount)
{
  return false;
}

CDavSocket*
CDavWorkSession::getSocket()
{
  return m_pSocket;
}

OI_RESULT 
CDavWorkSession::Connect(CDavRequest* pReq)
{
  OI_RESULT enuRet;
  OI_REQ_TYPE enuMethod = pReq->GetMethod();

  if(!pReq)
    return OIGEINVALIDARG;

  //handle Keep-Alive values
  time_t tmExpiresOn = m_tmConnectionTTL;
  if(tmExpiresOn != -1){
    time_t tmNow = time(NULL);
    //hard coded value
    if(tmNow -1 > tmExpiresOn){
      OI_DEBUG("Connection expired, disconnecting...\n");
      Disconnect();
    }
  }
  int nMaxConn = m_nMaxConnections;
  if(nMaxConn == 0){
    OI_DEBUG("Max connect reached, will disconnect after this request\n");
    SetPersistent(false);
  }

  //connect
  if(m_bUseProxy){
    switch(m_enuConnState){
    case S_CONN_DISCONNECTED:
      enuRet = m_pSocket->Connect(m_Proxy.GetHostName(), 
				  m_Proxy.GetPort());
      if(OI_OK != enuRet)
	return enuRet;
      m_enuConnState = S_CONN_CONNECTED;
      //falling through
    case S_CONN_CONNECTED:
      if(m_bUseSSL && (enuMethod != T_REQ_CONNECT)){
	enuRet = proxyTunnel();
	if(OI_OK != enuRet){
	  m_pSocket->Disconnect();
	  return enuRet;
	}
	enuRet = negotiateSSL();
	if(OI_OK != enuRet){
	  m_pSocket->Disconnect();
	  return enuRet;
	}
	m_enuConnState = S_CONN_TUNNELED;
      }
      break;
    default:
      break;
    }
  } else {
    switch(m_enuConnState){
    case S_CONN_DISCONNECTED:
      enuRet = m_pSocket->Connect(m_Server.GetHostName(), 
				  m_Server.GetPort());
      if(OI_OK != enuRet)
	return enuRet;
      if(m_bUseSSL){
	enuRet = negotiateSSL();
	if(OI_OK != enuRet){
	  m_pSocket->Disconnect();
	  return enuRet;
	}
      }
      m_enuConnState = S_CONN_CONNECTED;
      break;
    default:
      break;
    }
  }
  return OI_OK;
}

void 
CDavWorkSession::init()
{
  m_bExpect100 = false;
  m_bHttp11 = true;
  m_bPersistent = true;
#ifdef HAVE_LIBZ
  m_bUseCompression = false;
#endif /*HAVE_LIBZ*/
  m_bUseSSL = false;
  m_bUseProxy = false;
  m_tmConnectionTTL = -1;
  m_nMaxConnections = -1;
  m_unResponseLength = 0;
  m_ulPropsToParse = OI_RNP_CONVENTIONAL;
  m_enuConnState = S_CONN_DISCONNECTED;
  m_enuLastOperation = OP_USE_DEFAULT;

  m_pSocket = new CDavSocket(this);
  m_pAuthManager = new CDavAuthManager(this);
  m_pXmlBuffer = new CDavXmlBuffer;
  m_pMemBuffer = new CDavMemBuffer;
  m_pServerPolicy = new CDavServerPolicy;
  m_pUnauthenticCerts = new CDavSSLCertificateList();
  m_pCurrentLock = NULL;
  m_pActiveRequest = NULL;
}


OI_RESULT 
CDavWorkSession::proxyTunnel()
{
  bool bUseSSL = m_pSocket->UseSSL(false);
  OI_RESULT enuRet;
  OI_STRSTREAM_A ssURI;
  CDavRequest reqConnect, *pOldReq;
  CRBPVDDummy pvdDummy;
  CPBCSMDummy csmDummy(&reqConnect);

  ssURI << m_Server.GetHostName() << ":" << m_Server.GetPort();
  enuRet = reqConnect.Create(this,
			     T_REQ_CONNECT,
			     OI_METHOD_CONNECT,
			     ssURI.str().c_str());
  if(OI_OK != enuRet)
    return enuRet;
  
  /* save active request pointer */
  OI_ASSERT(m_pActiveRequest);
  pOldReq = m_pActiveRequest;

  m_pActiveRequest = NULL;
  enuRet = reqConnect.Dispatch(&pvdDummy, &csmDummy);

  /* restore the pointer */
  m_pActiveRequest = pOldReq;

  if(OI_OK != enuRet || reqConnect.GetStatusClass() != 2){
    m_pSocket->UseSSL(bUseSSL);
    return OIEETUNNELFAILED;
  }
  m_pSocket->UseSSL(true);
  return enuRet;
}

OI_RESULT 
CDavWorkSession::negotiateSSL()
{
  OI_RESULT enuRet = m_pSocket->connectSSL();
  if(OI_OK != enuRet)
    return enuRet;

  if(m_pUnauthenticCerts->GetCount() > 0){
    if(!OnVerifyCertificates(m_pUnauthenticCerts)){
      m_pSocket->cleanupSSL();
      m_pSocket->cleanupSSLSession();
      return OIEECERTCHECKFAILED;
    }
  }
  return OI_OK;
}

OI_SIZE_T 
CDavWorkSession::GetReadBytes()
{
  return m_pSocket->GetReadBytes();
}

unsigned int
CDavWorkSession::GetSocketTimeout()
{
  return m_pSocket->GetTimeout();
}

OI_SIZE_T 
CDavWorkSession::GetWrittenBytes()
{
  return m_pSocket->GetWrittenBytes();
}

CDavXmlBuffer* 
CDavWorkSession::GetXmlBuffer()
{
  return m_pXmlBuffer;
}

CDavMemBuffer* 
CDavWorkSession::GetMemBuffer()
{
  return m_pMemBuffer;
}

CDavSSLCertificateList* 
CDavWorkSession::GetUnauthenticCertificates()
{
  return m_pUnauthenticCerts;
}

static X509*
findIssuer(X509_STORE_CTX* pStoreCtx,
	   STACK_OF(X509)* pStack,
	   X509* pX509)
{
  int nIndex, nCount;
  X509* pIssuer;
  nCount = sk_X509_num(pStack);
  for(nIndex = 0; nIndex < nCount; nIndex++) {
    pIssuer = sk_X509_value(pStack, nIndex);
    if(pStoreCtx->check_issued(pStoreCtx, pX509, pIssuer))
      return pIssuer;
  }
  return NULL;
}

static bool
retriveFailures(CDavSSLCertificateList& certList,
		CDavSSLCertificate& cert)
{
  unsigned int unIndex, unCount;
  unCount = certList.GetCount();
  for(unIndex = 0; unIndex < unCount; unIndex++) {
    CDavSSLCertificate& certFailed = certList[unIndex];
    if(certFailed == cert) {
      cert.m_ulFailures = certFailed.m_ulFailures;
      return true;
    }
  }
  return false;
}

bool
CDavWorkSession::BuildCertificateChain(CDavSSLCertificateList* pOut)
{
  if(!pOut || !m_pSocket->HasSSLConnection())
    return false;

  pOut->Clear();
  SSL* pSSL = m_pSocket->getSSL();
  if(!pSSL)
    return false;

  STACK_OF(X509)* pPeerChain = SSL_get_peer_cert_chain(pSSL);
  X509* pPeer = SSL_get_peer_certificate(pSSL);
  if(!pPeerChain || !pPeer)
    return false;

  X509_STORE_CTX storeCtx;
  if(!X509_STORE_CTX_init(&storeCtx,
			  pSSL->ctx->cert_store,
			  pPeer,
			  pPeerChain))
    return false;

  /* first lookup certs in peer chain */
  X509* pIssuer;
  for(pIssuer = pPeer; pIssuer;) {
    CDavSSLCertificate curCert(pIssuer);
    retriveFailures(*m_pUnauthenticCerts, curCert);
    pOut->AddCertificate(curCert);
    /* have we reached root CA yet? */
    if(curCert.IsSelfSigned())
      return true;
    /* last konwn issuer is stored in pPeer */
    pIssuer = findIssuer(&storeCtx, pPeerChain, pIssuer);
    if(pIssuer)
      pPeer = pIssuer;
  }

  pIssuer = pPeer;
  /* next check against our cert store */
  for(;pIssuer;) {
    int nRet = storeCtx.get_issuer(&pIssuer, &storeCtx, pPeer);
    if(nRet <= 0) {
      /* issuer not found, but we're finished anyway */
      return true;
    }
    OI_ASSERT(pIssuer);
    pPeer = pIssuer;
    CDavSSLCertificate curCert(pIssuer);
    pOut->AddCertificate(curCert);
    /* have we reached root CA yet? */
    if(curCert.IsSelfSigned()) {
      return true;
    }
  }
  return true;
}


void 
CDavWorkSession::onResponseFound(CDavRequest* pReq, XNS(DOMNode) *pNode)
{
  //TODO:Implement this
  switch(pReq->GetMethod()){
  case T_REQ_PROPFIND:
    {
      CDavResourceNode theNode;
      theNode.Parse(pNode, m_ulPropsToParse);
      OnResourceFound(pReq, &theNode);
      break;
    }
  case T_REQ_LOCK:
    {
      if(m_pCurrentLock)
	CDavResourceNode::parseLockStatic(pNode, m_pCurrentLock);
      break;
    }
  case T_REQ_SEARCH:
    {
      OnSearchResponse(pReq, pNode);
      break;
    }
  default:
    OI_ASSERT(false);
    break;
  }
}


OI_RESULT 
CDavWorkSession::DoPropfind(const X& strURI, OI_PFIND_DEPTH enuDepth)
{
  OI_DEBUG("Starting PROPFIND on <%s>...\n", CX2CA(strURI));
  OI_RESULT enuRet;
  const char* pszDepth;

  switch(enuDepth){
  case D_PFIND_ZERO:
    pszDepth = "0";
    break;
  case D_PFIND_INFINITE:
    pszDepth = "infinity";
    break;
  case D_PFIND_ONE:
  default:
    pszDepth = "1";
    break;
  }
  CDavRequest reqPropfind;
  CPBCSMXml csmXml(&reqPropfind, OI_XML_CUTELEMENT);
  CRBPVDXmlPropfind pvdXmlPropfind;
  //TODO: implement more versions of this function
  reqPropfind.AddRequestHeader(OI_REQHDR_DEPTH, pszDepth);
  reqPropfind.AddRequestHeader(OI_REQHDR_CONTENTTYPE, "text/xml");
  enuRet =reqPropfind.Create(this, T_REQ_PROPFIND, OI_METHOD_PROPFIND, strURI);
  if(OI_OK != enuRet)
    return enuRet;
  enuRet = reqPropfind.Dispatch(&pvdXmlPropfind, &csmXml);
  return enuRet;
}

OI_RESULT 
CDavWorkSession::DoCopy(const X& strSrc,
			const X& strDest,
			CRBPVDXmlCopy* pXml,
			bool bOverwrite,
			const char* pszIfHeader)
{
  OI_DEBUG("Starting COPY from <%s> to <%s>...\n", CX2CA(strSrc), CX2CA(strDest));
  OI_RESULT enuRet;
  CDavRequest reqCopy;
  CRBPVDDummy pvdDummy;
  CRequestBodyProvider* pBodyProvider = &pvdDummy;
  CPBCSMDummy csmDummy(&reqCopy);
  OI_STRING_A strDestURI = Escape(((X&)strDest).UTF8());
  OI_STRING_A strOverwrite = bOverwrite ? "T" : "F"; 
  enuRet = reqCopy.Create(this, T_REQ_COPY, OI_METHOD_COPY, strSrc);
  if(OI_OK != enuRet)
    return enuRet;
  reqCopy.AddRequestHeader(OI_REQHDR_DESTINATION, strDestURI.c_str());
  reqCopy.AddRequestHeader(OI_REQHDR_OVERWRITE, strOverwrite.c_str());
  if(pXml){
    pBodyProvider = pXml;
    reqCopy.AddRequestHeader(OI_REQHDR_CONTENTTYPE, "application/xml");
  }

  if(pszIfHeader)
    reqCopy.AddRequestHeader(OI_REQHDR_IF, pszIfHeader);
  
  enuRet = reqCopy.Dispatch(pBodyProvider, &csmDummy);
  return enuRet;

}

OI_RESULT 
CDavWorkSession::DoDelete(const X& strURI,
			  const char* pszIfHeader)
{
  OI_DEBUG("Starting DELETE at <%s>...\n", CX2CA(strURI));
  OI_RESULT enuRet;
  CDavRequest reqDelete;
  CRBPVDDummy pvdDummy;
  CPBCSMDummy csmDummy(&reqDelete);
  enuRet = reqDelete.Create(this, T_REQ_DELETE, OI_METHOD_DELETE, strURI);
  if(OI_OK != enuRet)
    return enuRet;

  if(pszIfHeader)
    reqDelete.AddRequestHeader(OI_REQHDR_IF, pszIfHeader);

  enuRet = reqDelete.Dispatch(&pvdDummy, &csmDummy);
  return enuRet;
}

OI_RESULT 
CDavWorkSession::DoGet(const X& strURI,
		       const char* pszLocalPath,
		       OICRANGE* pRange)
{
  OI_RESULT enuRet;
  int nFlags = O_WRONLY | O_CREAT | O_TRUNC;
#ifdef WIN32
  nFlags |= O_BINARY;
#endif /* WIN32 */	
  int nHandle = open(pszLocalPath, nFlags); 
  if(nHandle == -1)
    return OIGEFILEOPENFAILED;
  enuRet = DoGet(strURI, nHandle, pRange);
  close(nHandle);
  return enuRet;
}

OI_RESULT 
CDavWorkSession::DoGet(const X& strURI,
		       int hFile,
		       OICRANGE* pRange)
{
  OI_DEBUG("Starting GET on <%s>...\n", CX2CA(strURI));
  OI_ASSERT(hFile != -1);

  OI_RESULT enuRet;
  CDavRequest reqGet;
  CRBPVDDummy pvdDummy;
  CPBCSMLocalFile csmFile(&reqGet, hFile);
  //reqGet.AddRequestHeader("User-Agent", "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0; YComp 5.0.0.0)");
  /* ranged get */
  if(pRange) {
    char chBuf[80];
    OI_SSIZE_T nStart, nEnd;
    nStart = pRange->nStart;
    nEnd = pRange->nEnd;

    if(nStart >= 0) {
      if(nEnd >= 0) {
	if(nEnd >= nStart) {
	  sprintf(chBuf, "bytes="OI_SIZEFORMAT"-"OI_SIZEFORMAT,
		  (OI_SIZE_T)nStart, (OI_SIZE_T)nEnd);
	  reqGet.AddRequestHeader("Range", chBuf);
	}
      } else {
	  sprintf(chBuf, "bytes="OI_SIZEFORMAT"-",(OI_SIZE_T)nStart);
	  reqGet.AddRequestHeader("Range", chBuf);
      }
    } else if(nEnd >= 0) {
      sprintf(chBuf, "bytes=-"OI_SIZEFORMAT, (OI_SIZE_T)nEnd);
      reqGet.AddRequestHeader("Range", chBuf);
    }
    reqGet.AddHandler(new CHandlerCRange(pRange));
  }

  enuRet = reqGet.Create(this, T_REQ_GET, OI_METHOD_GET, strURI);
  if(OI_OK != enuRet)
    return enuRet;

  return reqGet.Dispatch(&pvdDummy, &csmFile);
}

OI_RESULT 
CDavWorkSession::DoHead(const X& strURI)
{
  OI_DEBUG("Starting HEAD on <%s>...\n", CX2CA(strURI));
  OI_RESULT enuRet;
  CDavRequest reqHead;
  CRBPVDDummy pvdDummy;
  CPBCSMDummy csmDummy(&reqHead);
  enuRet = reqHead.Create(this, T_REQ_HEAD, OI_METHOD_HEAD, strURI);
  if(OI_OK != enuRet)
    return enuRet;

  enuRet = reqHead.Dispatch(&pvdDummy, &csmDummy);
  return enuRet;
}

OI_RESULT
CDavWorkSession::DoLock(const X& strURI,
			CDavLock* pLock,
			const char* pszIfHeader)
{
  OI_DEBUG("Starting LOCK on <%s>...\n", CX2CA(strURI));
  if(m_pCurrentLock)
    return OIGEINVALIDSTATE;

  OI_RESULT enuRet;
  CDavLock lockDefault, *pLockIn = NULL;
  unsigned long unMask;	
  CDavRequest reqLock;

  if(pszIfHeader)
    reqLock.AddRequestHeader(OI_REQHDR_IF, pszIfHeader);

  /* set output */
  m_pCurrentLock = pLock;

  /* set input */
  if(pLock){
    pLockIn = pLock;
  } else {
    pLockIn = &lockDefault;
    pLockIn->m_pOwner = NULL;
    pLockIn->m_unMask = 0;
  }

  unMask = pLockIn->m_unMask;

  /* make sure all fields are set */
  if(!(unMask & OI_LIVF_TYPE))
    pLockIn->m_enuType = T_LOCK_WRITE;
  if(!(unMask & OI_LIVF_SCOPE))
    pLockIn->m_enuScope = SCP_LOCK_EXCLUSIVE;
  if(!(unMask & OI_LIVF_DEPTH))
    pLockIn->m_enuDepth = D_LOCK_INFINITE;
  if(!(unMask & OI_LIVF_TIMEOUT))
    pLockIn->m_lTimeout = 0;
  if(!(unMask & OI_LIVF_OWNER))
    pLockIn->m_pOwner = NULL;

  OI_STRING_A strDepth;
  OI_STRING_A strTimeout;

  //setting up depth
  switch(pLockIn->m_enuDepth){
  case D_LOCK_ZERO:
    strDepth = "0";
    break;
  case D_LOCK_INFINITE:
  default:
    strDepth = "infinity";
    break;
  }
  reqLock.AddRequestHeader(OI_REQHDR_DEPTH, strDepth.c_str());
  //setting upt timeout
  if(pLockIn->m_lTimeout > 0){
    char chTimeout[30];
    sprintf(chTimeout, "Second-%ld", pLockIn->m_lTimeout);
    strTimeout = chTimeout;
  } else {
    strTimeout = "Infinite";
  }
  reqLock.AddRequestHeader(OI_REQHDR_TIMEOUT, strTimeout.c_str());
  reqLock.AddRequestHeader(OI_REQHDR_CONTENTTYPE, "text/xml");
  enuRet = reqLock.Create(this, T_REQ_LOCK, OI_METHOD_LOCK, strURI);
  if(OI_OK != enuRet)
    return enuRet;

  CRBPVDXmlLock pvdLock(pLockIn);
  //the cut element is "activelock"
  CPBCSMXml csmLock(&reqLock, OI_XML_CUTELEMENT_LOCK);
  enuRet = reqLock.Dispatch(&pvdLock, &csmLock);
  m_pCurrentLock = NULL;
  return enuRet;
}

OI_RESULT 
CDavWorkSession::DoMkcol(const X& strURI,
			 const char* pszIfHeader,
			 CRequestBodyProvider* pBodyProvider,
			 const char* pszContentType)
{
  OI_DEBUG("Starting MKCOL at <%s>...\n", CX2CA(strURI));
  OI_RESULT enuRet;
  CDavRequest reqMkcol;
  CRBPVDDummy pvdDummy;
  CPBCSMDummy csmDummy(&reqMkcol);
  enuRet = reqMkcol.Create(this, T_REQ_MKCOL, OI_METHOD_MKCOL, strURI);
  if(OI_OK != enuRet)
    return enuRet;
  if(pszContentType)
    reqMkcol.AddRequestHeader(OI_REQHDR_CONTENTTYPE, pszContentType);
  if(pszIfHeader)
    reqMkcol.AddRequestHeader(OI_REQHDR_IF, pszIfHeader);
  CRequestBodyProvider* pProvider = pBodyProvider ? pBodyProvider : &pvdDummy;
  enuRet = reqMkcol.Dispatch(pProvider, &csmDummy);
  return enuRet;
}

OI_RESULT 
CDavWorkSession::DoMove(const X& strSrc,
			const X& strDest,
			CRBPVDXmlMove* pXml,
			bool bOverwrite,
			const char* pszIfHeader)
{
  OI_DEBUG("Starting MOVE from <%s> to <%s>...\n", CX2CA(strSrc), CX2CA(strDest));
  OI_RESULT enuRet;
  CDavRequest reqMove;
  CRBPVDDummy pvdDummy;
  CRequestBodyProvider* pBodyProvider = &pvdDummy;
  CPBCSMDummy csmDummy(&reqMove);
  OI_STRING_A strDestURI = Escape(((X&)strDest).UTF8());
  OI_STRING_A strOverwrite = bOverwrite ? "T" : "F"; 
  enuRet = reqMove.Create(this, T_REQ_MOVE, OI_METHOD_MOVE, strSrc);
  if(OI_OK != enuRet)
    return enuRet;
  reqMove.AddRequestHeader(OI_REQHDR_DESTINATION, strDestURI.c_str());
  reqMove.AddRequestHeader(OI_REQHDR_OVERWRITE, strOverwrite.c_str());
  if(pXml){
    pBodyProvider = pXml;
    reqMove.AddRequestHeader(OI_REQHDR_CONTENTTYPE, "application/xml");
  }

  if(pszIfHeader)
    reqMove.AddRequestHeader(OI_REQHDR_IF, pszIfHeader);

  enuRet = reqMove.Dispatch(pBodyProvider, &csmDummy);
  return enuRet;
}

OI_RESULT
CDavWorkSession::DoOptions(const X& strURI,
			   OI_HEADERMAP* pHeaders)
{
  OI_DEBUG("Starting OPTIONS on <%s>...\n", CX2CA(strURI));
  OI_RESULT enuRet;
  CDavRequest reqOptions;
  CRBPVDDummy pvdDummy;
  CPBCSMDummy csmDummy(&reqOptions);

  enuRet = reqOptions.Create(this, T_REQ_OPTIONS, OI_METHOD_OPTIONS, strURI);
  if(OI_OK != enuRet)
    return enuRet;
  enuRet = reqOptions.Dispatch(&pvdDummy, &csmDummy);

  if(OI_OK == enuRet){
    if(pHeaders) {
      *pHeaders = reqOptions.m_mapRespHeaders;
    }
  }
  return enuRet;
}

OI_RESULT
CDavWorkSession::DoProppatch(const X& strURI,
			     CRBPVDXmlProppatch* pXml,
			     const char* pszIfHeader)
{
  OI_ASSERT(pXml);
  OI_DEBUG("Starting PROPPATCH on <%s>...\n", CX2CA(strURI));
  OI_RESULT enuRet;
  CDavRequest reqProppatch;
  CPBCSMXml csmProppatch(&reqProppatch, NULL); //we don't process in SAX mode

  enuRet = reqProppatch.Create(this, T_REQ_PROPPATCH, OI_METHOD_PROPPATCH, strURI);
  if(OI_OK != enuRet)
    return enuRet;
  reqProppatch.AddRequestHeader(OI_REQHDR_CONTENTTYPE, "application/xml");
  if(pszIfHeader)
    reqProppatch.AddRequestHeader(OI_REQHDR_IF, pszIfHeader);

  enuRet = reqProppatch.Dispatch(pXml, &csmProppatch);
  if(OI_OK == enuRet)
    m_pXmlBuffer->ImportDocument(csmProppatch.GetDocument());
  return enuRet;
}

OI_RESULT 
CDavWorkSession::DoPut(const X& strURI,
		       const char* pszLocalPath,
		       const char* pszIfHeader,
		       const char* pszContentType)
{
  OI_RESULT enuRet;
  int nFlags = O_RDONLY;
#ifdef WIN32
  nFlags |= O_BINARY;
#endif /* WIN32 */
  int nHandle = open(pszLocalPath, nFlags); 
  if(nHandle == -1)
    return OIGEFILEOPENFAILED;
  enuRet = DoPut(strURI, nHandle, pszIfHeader, pszContentType);
  close(nHandle);
  return enuRet;
}

OI_RESULT 
CDavWorkSession::DoPut(const X& strURI,
		       int hFile,
		       const char* pszIfHeader,
		       const char* pszContentType)
{
  OI_DEBUG("Starting PUT on <%s>...\n", CX2CA(strURI));
  OI_ASSERT(hFile != -1);

  OI_RESULT enuRet;
  CDavRequest reqPut;
  CRBPVDLocalFile pvdFile(hFile);
  CPBCSMDummy pvdDummy(&reqPut);
  //reqGet.AddRequestHeader("User-Agent", "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0; YComp 5.0.0.0)");
  enuRet = reqPut.Create(this, T_REQ_PUT, OI_METHOD_PUT, strURI);
  if(OI_OK != enuRet)
    return enuRet;
  if(pszIfHeader)
    reqPut.AddRequestHeader(OI_REQHDR_IF, pszIfHeader);

  if(pszContentType)
    reqPut.AddRequestHeader(OI_REQHDR_CONTENTTYPE, pszContentType);

  return reqPut.Dispatch(&pvdFile, &pvdDummy);
}

OI_RESULT
CDavWorkSession::DoSearch(const X& strURI,
			  CRBPVDXmlSearch* pXml)
{
  OI_ASSERT(pXml);
  OI_DEBUG("Starting SEARCH on <%s>...\n", CX2CA(strURI));
  OI_RESULT enuRet;
  CDavRequest reqSearch;
  CPBCSMXml csmSearch(&reqSearch, OI_XML_CUTELEMENT);

  enuRet = reqSearch.Create(this, T_REQ_SEARCH, OI_METHOD_SEARCH, strURI);
  if(OI_OK != enuRet)
    return enuRet;
  reqSearch.AddRequestHeader(OI_REQHDR_CONTENTTYPE, "application/xml");

  enuRet = reqSearch.Dispatch(pXml, &csmSearch);
  return enuRet;

}

OI_RESULT
CDavWorkSession::DoUnlock(const X& strURI,
			  const char* pszLockToken)
{
  OI_DEBUG("Starting UNLOCK on <%s>...\n", CX2CA(strURI));
  OI_ASSERT(pszLockToken);
  OI_RESULT enuRet;
  OI_STRING_A strLockToken = "<";
  strLockToken += pszLockToken;
  strLockToken += ">";
  CDavRequest reqUnlock;
  CRBPVDDummy pvdDummy;
  CPBCSMDummy csmDummy(&reqUnlock);

  enuRet = reqUnlock.Create(this, T_REQ_UNLOCK, OI_METHOD_UNLOCK, strURI);
  if(OI_OK != enuRet)
    return enuRet;
  reqUnlock.AddRequestHeader(OI_REQHDR_LOCKTOKEN, strLockToken.c_str());

  enuRet = reqUnlock.Dispatch(&pvdDummy, &csmDummy);
  return enuRet;
}

OI_RESULT
CDavWorkSession::CreateStream(CDavHttpStream** ppStream,
			      const X& strURI,
			      OI_STREAM_MODE enuMode)
{
  OI_RESULT enuRet;
  CDavHttpStream* pStream = new CDavHttpStream;
  enuRet = pStream->createStream(this, strURI, enuMode);
  if(OI_OK != enuRet) {
    delete pStream;
    pStream = NULL;
  }
  *ppStream = pStream;
  return enuRet;
}

OI_RESULT
CDavWorkSession::DestroyStream(CDavHttpStream* pStream)
{
  if(!pStream || (pStream != m_pActiveRequest))
    return OIGEINVALIDARG;
  /*stream will  Disconnect() if transfer is unfinished*/ 
 delete pStream;
 return OI_OK;
}

CDavServerPolicy* 
CDavWorkSession::GetServerPolicy()
{
  return  m_pServerPolicy;
}

CDavAuthManager*
CDavWorkSession::GetAuthManager()
{
  return m_pAuthManager;
}

/*
 * Add an unauthentic cert to the list
 * pX509 should not be incremented before calling this function
 * if the same cert already exists in the list, the failure flag bits
 * will be "OR"ed to the original
 */
void 
CDavWorkSession::addUnauthenticCert(X509* pX509, unsigned long ulFailures)
{
  if(!pX509 || !ulFailures)
    return;
  //first seach through the list, see if there is a match
  int nCount = m_pUnauthenticCerts->GetCount();
  for(int nIndex = 0; nIndex < nCount; nIndex++){
    CDavSSLCertificate& cert = (*m_pUnauthenticCerts)[nIndex];
    if(cert == pX509)
      cert.m_ulFailures |= ulFailures;
    return;
  }
  //no match, add the cert to the list
  m_pUnauthenticCerts->AddCertificate(CDavSSLCertificate(pX509, ulFailures));
}


