/* 
   Dav session handler
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 1999-2003, Joe Orton <joe@manyfish.co.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavWorkSession.h 132 2005-06-24 09:09:43Z komat $
*/

#ifndef ONION_HEADER_DAVWORKSESSION_H
#define ONION_HEADER_DAVWORKSESSION_H 1

#include <onion/OnionCfg.h>
#include <onion/DavHost.h>
#include <onion/DavRequest.h>
#include <xercesc/dom/DOM.hpp>
#include <openssl/x509.h>
#include <time.h>

//forward declarations
class CDavSocket;
class CDavLock;
class CDavServerPolicy;
class CDavResourceNode;
class CDavAuthManager;
class CDavHttpStream;
class CDavMemBuffer;
class CDavXmlBuffer;
class CDavSSLCertificateList;
class CDavSSLClientCertificate;
class CDavSSLDName;
class CRBPVDXmlCopy;
class CRBPVDXmlMove;
class CRBPVDXmlProppatch;
class CRBPVDXmlSearch;
class CRequestBodyProvider;
class CResponseBodyConsumer;

/**This class is the primary interface of onion library.
 *
 * CDavWorkSession is designed to complete most of the client tasks
 * with one function call. To get a simple application into action,
 * usually overriding 1 or 2 handler is enough, depending on the
 * purpose of the program.
 *
 * In case you want to have finer control over the whole process, or
 * if you want to change the way libonion behaves, e.g. providing
 * client certificate on demand, showing progress information, adding
 * your custom header, etc., you'll need to override more virtual
 * functions.
 *
 */
OI_BEGIN_CLASSEXPORT

class OI_EXPORT CDavWorkSession
{
  //friends
  friend class CDavHttpStream;
  friend class CDavRequest;
  friend class CDavSocket;
  friend class CPBCSMXml;
 public:

  /** @name Constructor */
  /* @{ */
  /**Constructor
   */
  CDavWorkSession();
  /* @} */

  /** @name Destructor */
  /* @{ */
  /**Destructor
   */
  virtual ~CDavWorkSession();
  /* @} */


  /** @name DAV methods */
  /* @{ */

  //dav methods
  /**method COPY
   *
   * This function copies resource identified by \e strSrc to \e strDest.
   * An XML body may be provided for the operation.
   *
   * @param strSrc source URI
   * @param strDest destination URI
   * @param pXml optional XML body
   * @param bOverwrite whether to overwrite if destination exists
   * @param pszIfHeader the 'If' header to justify overwriting
   *
   * @return
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   * @note Source and destination URIs are transcoded to UTF8 and escaped
   *	automatically. However, \e 'if' header is not. So you'll have to do
   *	it yourself by calling methods of CDavXmlString.
   */
  OI_RESULT DoCopy(const X& strSrc,
		   const X& strDest,
		   CRBPVDXmlCopy* pXml = NULL,
		   bool bOverwrite = false,
		   const char* pszIfHeader = NULL);

  /**method DELETE
   *
   * This function deletes resource identified by \e strURI.
   *
   * @param strURI URI of the resource to be deleted
   * @param pszIfHeader the 'If' header to justify deleting
   *
   * @return
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   * @note URIs are transcoded to UTF8 and escaped automatically. However,
   *	\e 'if' header is not. So you'll have to do it yourself by calling
   *	methods of CDavXmlString.
   */
  OI_RESULT DoDelete(const X& strURI,
		     const char* pszIfHeader = NULL);

  /**method GET
   *
   * This function download a  resource identified by \e strURI and writes
   * it to file identified by \e hFile. Range can be specified for this
   * method to get part of a resource.
   * \par How to use ranged get
   *	Content-Range is specified by setting the \e nStart and \e nEnd
   *	field of a OICRANGE structure. By setting either \e nStart or
   *	\e nEnd to negative, you can have the server to use default range
   *	either from \b BOF or from \b EOF. For example, a value pair of
   *	(4, -1) will cause server to transfer from the 4th byte until
   *	\b EOF, and (-1, 4) will cause server to transfer the \b LAST
   *	\b 4 \b BYTES.\n
   *	Ranged get result will be returned in the same structure using
   *	fields \e nStart, \e nEnd and \e nTotal.
   *
   * @param strURI URI of the resource to be downloaded
   * @param hFile handle of the file to be written to
   * @param pRange pointer to a OICRANGE struct to specify content
   *	range(use NULL to ignore range)
   *
   * @return an error code
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   */
  OI_RESULT DoGet(const X& strURI,
		  int hFile,
		  OICRANGE* pRange = NULL);

  /**method GET
   *
   * \overload 
   *
   * @param strURI URI of the resource to be downloaded
   * @param pszLocalPath local path for the file to be downloaded to 
   * @param pRange pointer to a OICRANGE struct to specify content
   *	range(use NULL to ignore range)
   *
   * @return an error code
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   */
  OI_RESULT DoGet(const X& strURI,
		  const char* pszLocalPath,
		  OICRANGE* pRange = NULL);

  /**method HEAD
   *
   * According to \b RFC2068:
   * \par
   * "The HEAD method is identical to GET except that the server
   * MUST NOT return a message-body in the response."
   *
   * It seems HEAD is not of much use and often problematic
   *
   * @param strURI URI of the resource
   *
   * @return an error code
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   */
  OI_RESULT DoHead(const X& strURI);

  /**method LOCK
   *
   * This function locks a DAV resource identified by /e strURI.
   * Lock parameters is passed in and out via \e pLock.
   * \b RFC2518 says:
   *
   * \par
   * The owner XML element provides information sufficient
   * for either directly contacting a principal (such as a telephone
   * number or Email URI), or for discovering the principal (such as the
   * URL of a homepage) who owns a lock.
   *
   * and
   *
   * \par
   * <!ELEMENT owner ANY>
   *
   * This means the owner XML element can be anything. Notice that libonion
   * supports string only.
   *
   * @param strURI URI of the resource to be locked
   * @param pLock pointer(in/out) to lock parameters, use NULL for default
   * @param pszIfHeader the 'If' header to justify the lock
   *
   * @return an error code
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   */
  OI_RESULT DoLock(const X& strURI,
		   CDavLock* pLock = NULL,
		   const char* pszIfHeader = NULL);

  /**method MKCOL
   *
   * The MKCOL method is used to create a collection resource at
   * specified URI. 'If' header and body can be used as well.
   *
   * @param strURI URI of the resource to be created
   * @param pszIfHeader 'if' header to justfy overwrite
   * @param pBodyProvider optional body provider
   * @param pszContentType pointer to content type in case a body is used
   *
   * @return an error code
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   */
  OI_RESULT DoMkcol(const X& strURI,
		    const char* pszIfHeader = NULL,
		    CRequestBodyProvider* pBodyProvider = NULL,
		    const char* pszContentType = NULL);

  /**method MOVE
   *
   * This function moves resource identified by \e strSrc to \e strDest.
   * An XML body may be provided for the operation.
   *
   * @param strSrc source URI
   * @param strDest destination URI
   * @param pXml optional XML body
   * @param bOverwrite whether to overwrite if destination exists
   * @param pszIfHeader the 'If' header to justify overwriting
   *
   * @return
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   * @note Source and destination URIs are transcoded to UTF8 and escaped
   *	automatically. However, \e 'if' header is not. So you'll have to do
   *	it yourself by calling methods of CDavXmlString.
   */
  OI_RESULT DoMove(const X& strSrc,
		   const X& strDest,
		   CRBPVDXmlMove* pXml = NULL,
		   bool bOverwrite = false,
		   const char* pszIfHeader = NULL);

  /**method OPTIONS
   *
   * Purpose of OPTIONS is to determine what operations are available
   * on the given URI. Notice the URI does not have to exist! Results
   * are usually returned in headers.
   *
   * @param strURI URI of the OPTIONS request
   * @param pHeaders (optional)pointer to a header map to duplicate all
   *	response headers to.
   *
   * @return
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   */
  OI_RESULT DoOptions(const X& strURI,
		      OI_HEADERMAP* pHeaders);
  /**method PROPFIND
   *
   * PROPFIND is used to list resource in a collection, or to list
   * propertie(s) of specified resource(s). It is quite similiar to \c
   * ls command on UNIX or \c dir command on DOS.
   *
   * @param strURI URI of the resource
   * @param enuDepth depth option, use one of the following:
   *	- \e D_PFIND_ZERO to list properties on this resource only
   *	- \e D_PFIND_ONE to list 1 level under this resource(\e default)
   *	- \e D_PFIND_INFINITE to list all sub resources and properties
   *
   * @return
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   */
  OI_RESULT DoPropfind(const X& strURI,
		       OI_PFIND_DEPTH enuDepth = D_PFIND_ONE);

  /**method PROPPATCH
   *
   * PROPPATCH shall be used when you want to set properties of
   * certain resource. To do this, an XML body is always reqired.
   *
   * @param strURI URI of the resource
   * @param pXml pointer to the XML body provider
   * @param pointer to 'If' header to justify overwriting
   *
   * @return
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   */
  OI_RESULT DoProppatch(const X& strURI,
			CRBPVDXmlProppatch* pXml,
			const char* pszIfHeader = NULL);


  /**method PUT
   *
   * PUT uploads a file to the specified URI. This is the file
   * handle version.
   *
   * @param strURI URI of the resource
   * @param hFile handler to the file to be uploaded
   * @param pszIfHeader (optional) pointer to 'If' header to justify
   *	overwriting
   * @param pszContentType (optinal) pointer to content type string
   *
   * @return
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   */
  OI_RESULT DoPut(const X& strURI,
		  int hFile,
		  const char* pszIfHeader = NULL,
		  const char* pszContentType = NULL);

  /**method PUT
   *
   * \overload
   *
   * @param strURI URI of the resource
   * @param pszLocalPath path to the file to be uploaded
   * @param pszIfHeader (optional) pointer to 'If' header to justify
   *	overwriting
   * @param pszContentType (optinal) pointer to content type string
   *
   * @return
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   */
  OI_RESULT DoPut(const X& strURI,
		  const char* pszLocalPath,
		  const char* pszIfHeader = NULL,
		  const char* pszContentType = NULL);

  /**method SEARCH
   *
   * SERACH method is similiar to PROPFIND except that both request
   * and response contents are 
   *
   * @param strURI URI of the resource
   * @param pXml pointer to the XML body provider
   *
   * @return
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   */
  OI_RESULT DoSearch(const X& strURI,
		     CRBPVDXmlSearch* pXml);

  /**method UNLOCK
   *
   * This is the opposite of LOCK method. However, to unlock a
   * resource, lock token of that resource is needed.
   *
   * @param strURI URI of the resource to be unlocked
   * @param pszLockToken pointer to string of the lock token
   *
   * @return an error code
   *	- \c OI_OK if successful
   *	- other value if error(s) occured
   *
   * @see CDavLock
   * @see #DoLock
   */
  OI_RESULT DoUnlock(const X& strURI,
		     const char* pszLockToken);
  /* @} */

  /**Is this session expecting 100 Continue ?
   *
   * This is not of much use nowadays.
   *
   * @return 'true' if the session is expecting 100 Continue,
   *	'false' otherwise.
   *
   * @see #SetExpecting100
   */
  bool IsExpecting100();

  /**Is this session persistent ?
   *
   * HTTP 1.1 allows connection to extend through multiple requests so
   * that client and server don't have to connect and disconnect every
   * time. But this depends on various factors. First of course is
   * whether the server supports HTTP 1.1 and whether you ask session
   * to try persistent connection. Second, it depends on whether
   * server is configured to allow persistent connection. Third, this
   * also depends on type of request you're issuing and current state
   * of server.
   *
   * Generally speaking, if you set persistent connection to
   * true(default) and issue a request for the first time, or if the
   * server returned a \e Connection: \e Keep-Alive header in the last
   * request. The return value will be true. But don't expect it to be
   * stable.
   *
   * @return 'true' if the session is persistent, 'false' otherwise.
   *
   * @see #SetPersistent
   */
  bool IsPersistent();


  /**Is last response from server in HTTP 1.1?
   *
   * @return 'true' if no response has been received yet or 
   *	if last response is in HTTP 1.1, 'false' otherwise.
   *
   * @see #SetHttp11
   */
  bool IsHttp11();

#ifdef HAVE_LIBZ

  /**Set the 'try compression' flag
   *
   *	HTTP 1.1 allows request and response body to be compressed to
   *	reduce network traffic(in most cases, of downloading). To
   *	accomplish that, negotiation of whether to use compression and
   *	which scheme to use needs to be commenced. So this does not
   *	necessarily guarentee compression. Moreover, compression also
   *	brings a considerable performance penalty to server. So
   *	whether to apply it to your contents should be evaluated case
   *	by case.  libonion implements only gzip compression for
   *	downloading for the moment.
   *
   * @param bValue The value whether to try compression or not.
   *
   * \attention This method is only available when zlib support is
   * enabled @see #IsUsingCompression
   */
  void SetUseCompression(bool bValue);

  /**Is compression enabled?
   *
   * @return 'true' if compression is enabled, 'false' otherwise.
   *
   * @see #SetUseCompression
   */
  bool IsUsingCompression();

#endif /*HAVE_LIBZ*/

  /**Is session using SSL connection?
   *
   * @return 'true' if SSL is being used, 'false' otherwise.
   *
   * @see #SetUseSSL
   */
  bool IsUsingSSL();

  /**Is session using proxy?
   *
   * @return 'true' if SSL is using proxy, 'false' otherwise.
   *
   * @see #SetUseProxy
   * @see #CDavSession::SetProxy
   */
  bool IsUsingProxy();

  /**Get raw read bytes
   *
   * This function returns network download traffic in bytes. It does
   * not equal to bytes read from session since this value include all
   * headers and other strings, and compression is not applied here.
   *
   * @return raw bytes read from socket since this session is created.
   */
  OI_SIZE_T GetReadBytes();

  /**Get socket timeout setting in seconds
   *
   * @return socket timeout setting in seconds
   */
  unsigned int GetSocketTimeout();

  /**Get raw written bytes
   *
   * This function returns network upload traffic in bytes. It does
   * not equal to bytes sent from session since this value include all
   * headers and other strings
   *
   * @return raw bytes sent from socket since this session is created.
   */
  OI_SIZE_T GetWrittenBytes();

  /**Get server policy pointer
   *
   * @return pointer(not NULL) to server policy of this session
   */
  CDavServerPolicy* GetServerPolicy();

  /**Get authentication manager pointer
   *
   * @return pointer(not NULL) to authentication manager of this session
   */
  CDavAuthManager* GetAuthManager();


  /**Set the 'expecting 100 Continue' flag
   *
   * @param bValue The value whether expect 100 is enabled.
   *
   * @see #IsExpecting100
   */
  void SetExpecting100(bool bValue);

  /**Set the 'HTTP 1.1' flag
   *
   * @param bValue The value whether issue request in HTTP 1.1
   *
   * @see #IsHttp11
   */
  void SetHttp11(bool bValue);

  /**Set the 'persistent' flag
   *
   * @param bValue The value whether try to make persistent connection
   *
   * @see #IsPersistent
   */
  void SetPersistent(bool bValue);

  /**Set socket timeout setting in seconds
   *
   * @param unTimeout The value of socket timeout setting in seconds
   *
   * @see #GetSocketTimeout
   */
  void SetSocketTimeout(unsigned int unTimeout);

  /**Set the 'use ssl' flag
   *
   * @param bValue The value whether try to use ssl connection
   *
   * @see #IsUsingSSL
   */
  void SetUseSSL(bool bValue);

  /**Set the 'use proxy' flag
   *
   * @param bValue The value whether try to use ssl proxy
   *
   * @see #IsUsingProxy
   */
  void SetUseProxy(bool bValue);
  
  /**Ensure a valid connection to server for the given request
   *
   * @param pReq Pointer to the request about to be issued
   *
   * This function ensures a valid connection for the given
   * request. If necessary, disconnect and re-connect. As these are
   * supposed to be handled by session automatically, you don't need
   * to call this function directly.
   *
   * @return error code
   *	- \e OI_OK if operation successful
   *	- \e OIGEINVALIDARG if pReq is not valid
   *	- \e OISEHOSTNOTFOUND if server or proxy is not found
   *	- \e OISEINVALIDSOCK if socket creation failed
   *	- \e OISETIMEOUT if timeout when trying to connect to server
   *	- \e OISECONNFAILED if connection was refused
   * @see #Disconnect
   */
  OI_RESULT Connect(CDavRequest* pReq);

  /**Disconnect from server immediately
   *
   * As connect and disconnect is automatically handled by session,
   * you don't need to call this function explicitly.  Unless of
   * course, you want to make sure session is disconnected.
   *
   * @see #Connect
   */
  void Disconnect();

  /**Get connection state
   *
   * This function will return last successful connection state.
   * Don't expect it to reflect precise realtime state because the
   * state flags is only operated by Connect() and Disconnect()
   * functions.
   *
   * @return connection state
   *
   * @deprecated this function is deprecated
   */
  OI_CONN_STATE GetConnectState();

  /**Get XML buffer pointer
   *
   * @return pointer(not NULL) to XML buffer of this session
   */
  CDavXmlBuffer* GetXmlBuffer();


  /**Get memory buffer pointer
   *
   * @return pointer(not NULL) to memory buffer of this session
   */
  CDavMemBuffer* GetMemBuffer();

  /**Get unauthentic certificate list pointer
   *
   * If session is connecting to server using SSL and some
   * certificate(s) from server did not pass verification, they will
   * be duplicated and insert into this list.
   *
   * @return pointer(not NULL) to unauthentic certificate list
   * of this session
   *
   * @see CDavSSLCertificateList
   */
  CDavSSLCertificateList* GetUnauthenticCertificates();

  /**Build a certificate chain from peer certificate
   *
   * @param pOut pointer to a certificate list for output
   *
   * @return 'false' if session is not connected to server
   *	via SSL, or error occured while initializing certificate
   *	store, 'true' otherwise.
   */
  bool BuildCertificateChain(CDavSSLCertificateList* pOut);


  /** @name Overrides */
  /* @{ */

  /**Authentication handler
   *
   * This function must be overriden if you want to use
   * authentication.  It will be called each time server asks for new
   * credentials.
   *
   * @param strRealm (in) realm given by server
   * @param strUsername (out) username
   * @param strPasswd (out) password
   * @param nRetry (in) retry count since last successful access
   * @param enuAuthClass (in) target of this authentication, server
   * or proxy
   *
   * @return 'true' if you want to try authenticating,
   *	'false' to give up
   */
  virtual bool OnAuthentication(X& strRealm,
				OI_STRING_A& strUsername,
				OI_STRING_A& strPasswd,
				int nRetry,
				OI_AUTH_CLASS enuAuthClass);

  /**Cancel send handler
   *
   * This function will be called each time request body sending
   * progress is canceled
   *
   * @param pReq pointer to the request
   * @param pRBProvider pointer to the provider of request body whose
   *	sending process is canceled
   *
   */
  virtual void OnCancelSend(CDavRequest* pReq,
			    CRequestBodyProvider* pRBProvider);

  /**Cancel reveive handler
   *
   * This function will be called each time response body receiving
   * progress is canceled
   *
   * @param pReq pointer to the request
   * @param pRBProvider pointer to the consumer of response body whose
   *	receiving process is canceled
   *
   */
  virtual void OnCancelReceive(CDavRequest* pReq,
			       CResponseBodyConsumer* pPBConsumer);

  //override for request
  /**Create request handler
   *
   * This function will be called each time a request is created
   *
   * @param pReq pointer to the request
   *
   */
  virtual void OnCreateRequest(CDavRequest* pReq);

  /**Build request handler
   *
   * This function will be called each time a request is about to be
   * built. This is the best place to make last-minute change to
   * request headers.
   *
   * @param pReq pointer to the request
   *
   */
  virtual void OnPreBuildRequest(CDavRequest* pReq);


  /**Pre send request handler
   *
   * This function will be called before sending request body.
   *
   * @param pReq pointer to the request
   *
   */
  virtual void OnPreSendRequest(CDavRequest* pReq);

  /**Post send request handler
   *
   * This function will be called after request body is sent.
   *
   * @param pReq pointer to the request
   *
   */
  virtual void OnPostSendRequest(CDavRequest* pReq);

  /**Pre receive response handler
   *
   * This function will be called before receiving response body.
   *
   * @param pReq pointer to the request
   *
   */
  virtual void OnPreRecvResponse(CDavRequest* pReq);

  /**Post receive response handler
   *
   * This function will be called after receiving response body.
   *
   * @param pReq pointer to the request
   *
   */
  virtual void OnPostRecvResponse(CDavRequest* pReq);

  /**Destroy request handler
   *
   * This function will be called each time a request object is
   * destroyed.
   *
   * @param pReq pointer to the request
   *
   */
  virtual void OnDestroyRequest(CDavRequest* pReq);

  /**Extra header process handler
   *
   * libonion processes all response headers by calling handlers.
   * Each time a header is processed, library will first check handler
   * map to see if there is a corresponding handler for the header. If
   * a handler is found, library will pass header values to the
   * handler. Otherwise this handler function will be called. This
   * gives you an alternative way to process custom response headers
   * without writing a header handler and add it to request object
   * everytime.
   *
   * @param pReq pointer to the request
   * @param pszName pointer to the header name string
   * @param vecValues an array of header values as OI_STRING_A
   *
   * @return an error code
   *
   * @note Any return code other than \c OI_OK will terminate the request
   */
  virtual OI_RESULT ProcessExtraHeader(CDavRequest* pReq,
				       const char* pszName,
				       OI_STRLIST_A& vecValues);

  /**Server string match handler
   *
   * libonion will try to identify the server before consulting server
   * policy for what to do. By default this is done by comparing \e
   * server response header string and \e Name attribute of each \e
   * Server node in server policy configuration. If 2 strings match
   * perfectly, config for this server is considered to be 'found',
   * otherwise library will go to next node and try to match the
   * strings again. If no match is found after all \e Server node is
   * enumerated, default config will be used. By overriding this
   * handler function, you can implement your own matching logic,
   * e.g. the first 10 characters.
   *
   * @param pszHeaderValue pointer to the string value of \e Server header
   * @param pszConfigValue pointer to server name string found in our config
   *
   * @return 'true' to indicate a match, 'false' otherwise
   *
   */
  virtual bool MatchServerString(const char* pszHeaderValue,
				 const char* pszConfigValue);
  //progress callbacks
  /**Send progress handler
   *
   * This handler function will be called each time a block of request body
   * is about to be sent. This is the right place for your UI progress update
   * and cancel code.
   *
   * @param pReq pointer to the request
   * @param unProgress byte count of body that has already been sent
   * @param unTotal totoal byte count of the body
   *
   * @return 'true' unless you want to cancel sending progress.
   *
   */
  virtual bool OnSendProgress(CDavRequest* pReq,
			      OI_SIZE_T unProgress,
			      OI_SIZE_T unTotal);

  /**Receive progress handler
   *
   * This handler function will be called each time a block of
   * response body is received and about to be processed. This is the
   * right place for your UI progress update and cancel code. Notice
   * that we won't be able to konw body total length every time unless
   * server always sends \e Content-Length header.So the unTotal
   * parameter is only valid when bCLength is 'true'.
   *
   * @param pReq pointer to the request
   * @param unProgress byte count of body that has already been sent
   * @param unTotal totoal byte count of the body
   * @param bCLength whether unTotal is valid
   *
   * @return 'true' unless you want to cancel sending progress.
   *
   * @note Whether, when and how this handler is called actually
   * depends on response body consumers. see #response_body_consumers
   * for more details.
   *
   */
  virtual bool OnReceiveProgress(CDavRequest* pReq,
				 OI_SIZE_T unProgress,
				 OI_SIZE_T unTotal,
				 bool bCLength);

  /**DAV resource handler for \c PROPFIND
   *
   * This handler function will be called each time a complete DAV
   * resource node is sent to client and parsed without
   * errors. Properties defined in \b RFC2518, like getcontentlength,
   * displayname, etc. are parsed automatically. Other custom
   * properties are stored in 2 DOM nodes.  Contents of the node will
   * be discarded to limit memory usage. So you'll have to copy
   * information here is you want to save it.
   *
   * @param pReq pointer to the request
   * @param pNode pointer to the DAV resource node
   *
   * @note This handler will only be called during \c PROPFIND.
   *
   * @see CDavResourceNode
   */
  virtual void OnResourceFound(CDavRequest* pReq, CDavResourceNode* pNode);

  /**DAV serch response handler
   *
   * This handler function is similiar to OnResponseFound except this is for
   * \c SEARCH method and all contents are presented as a DOM node.
   *
   * @param pReq pointer to the request
   * @param pNode pointer to the DOMNode object
   *
   */
  virtual void OnSearchResponse(CDavRequest* pReq, XNS(DOMNode)* pNode);

  //SSL callbacks
  /**SSL certificate verification handler
   *
   * This handler function will be called when library is trying to establish
   * a SSL connection with server and some(or all) of the certificate(s) that
   * server presented is problematic. Problem bits can be retrived from
   * CDavSSLCertificate::m_ulFailures. Connecting to a server with unauthentic
   * certificate is not safe, so usually you should leave the choice to the
   * user.
   *
   * @param pUnauthenticCerts pointer(not NULL) to an array of problematic 
   *	certificates
   *
   * @return 'true' to omit the problem and go on connecting to the server,
   *	'false' will fail the connection
   */
  virtual bool OnVerifyCertificates(CDavSSLCertificateList* pUnauthenticCerts);

  /**Provede client certificate handler
   *
   * This handler function will be called when you did not load a client
   * certificate explicitly using LoadClientCertificate() and a client
   * certificate is required by the server.
   *
   * @param pCertOut pointer to the client certificate to load to
   * @param pCAList pointer to acceptable CAs given by server, count is given
   *	in nCount
   * @param nCount count of CAs given by server
   *
   * @note certificate provided through this function will disppear each
   *	time the connection is lost. So client certificate may be requested
   *	when reconnecting to server.
   */
  virtual bool OnProvideClientCertificate(CDavSSLClientCertificate* pCertOut,
					  CDavSSLDName* pCAList,
					  int nCount);

  /* @} */

  /** @name Stream Operations */
  /* @{ */

  /**Create a stream from a DAV resource
   *
   * Instead of writing resource contents to local file or buffer in
   * 'push' mode, libonion can also offer a 'pull' mode to let client
   * programs control read process. Currently stream is read only and
   * any error during transmission will also terminate the stream.
   * Thus if you create a stream and wait a long time before starting
   * to read, it may timeout and return an error code before actually
   * reading any data. Whether or not this happends actually depends
   * on a lot of things, especially server configuration. So it'll be
   * wise not to think this stream is reliable and never treat it like
   * a local file stream.\n
   *
   * Another thing that needs your attention is that, at any time,
   * only one stream can be 'active' for a certain session. Creating a
   * second stream or issuing another DAV operation without destroying
   * the previous active stream will cause the session to do nothing
   * but return an \e OIGEINVALIDSTATE error code. libonion does not
   * mess with multi-thread issues because that's irrelevant here. And
   * remember, destroying a stream without reading until \b EOF will
   * disconnect session from server. This is quite natrual if you
   * consider the question in the term of comminication between server
   * and client.
   *
   * @param ppStream (out) address of a pointer variable to receive the
   *	stream
   * @param strURI URI of the resource to be read from
   * @param enuMode stream mode, only \e M_STR_READ is supported
   *
   * @return an error code
   *	- \e OI_OK if successful
   *	- \e OIGENOTIMPLEMENTED if enuMode is not \e M_STR_READ
   *	- other value if any other error(s) occured
   *
   * @see CDavHttpStream
   * @see #DestroyStream
   *
   */
  OI_RESULT CreateStream(CDavHttpStream** ppStream,
			 const X& strURI,
			 OI_STREAM_MODE enuMode);

  /**Destroy an active stream
   *
   * This function destroys active stream created with #CreateStream.
   * Only the session that created the stream can destroy it. If data
   * transmission is not yet finished, session will be disconnected
   * from server.
   *
   * @param pStream pointer to the stream to be destroyed
   *
   * @return an error code
   *	- \e OI_OK if successful
   *	- other value if any other error(s) occured
   *
   * @see CDavHttpStream
   * @see #CreateStream
   *
   *
   */
  OI_RESULT DestroyStream(CDavHttpStream* pStream);

  /* @} */

 protected:
  CDavSocket*	getSocket();
  void addUnauthenticCert(X509* pX509, unsigned long ulFailures);
  void onResponseFound(CDavRequest* pReq, XNS(DOMNode)* pNode);
 private:
  void	init();
  OI_RESULT proxyTunnel();
  OI_RESULT negotiateSSL();
 public:

  /** server
   *
   * server information
   *
   */
  CDavHost m_Server;

  /** proxy
   *
   * proxy information
   * 
   */
  CDavHost m_Proxy;

  /**last status code
   *
   * HTTP status code for last request dispatched. -1 if no request
   * has been dispatched yet
   */
  int m_nLastStatusCode;

  /**max connection count indicator
   *
   * In how many requests will current connection expire (-1 means
   * never)
   */
  int m_nMaxConnections;

  /**connection expire time indicator
   *
   * when will current connection expire (-1 means never)
   */
  time_t m_tmConnectionTTL;

  /**user agent string
   *
   * string value of the \e User-Agent header
   */
  OI_STRING_A m_strUserAgent;

  /**last server string
   *
   * string value of the first \e Server header of last request
   * dispatched.
   */
  OI_STRING_A m_strServerString;

  /**last location
   *
   * location string from \e Location header of last request
   */
  OI_STRING_A m_strLocation;

  /**error description string
   *
   * error description(only for xml exception so far)
   */
  OI_STRING_A m_strErrorDescription;

  /**real body length counter
   *
   * uncompressed response body length byte counter, provided for
   * convenience
   */
  OI_SIZE_T m_unResponseLength;

  /**last OP code
   *
   * last operation server executed
   */
  OI_OPCODE m_enuLastOperation;

  /**PROPFIND parse flags
   *
   * which properties to parse while doing PROPFIND
   */
  unsigned long m_ulPropsToParse;

 protected:
  /**request being dispatched
   *
   * this pointer will become valid when a request is being dispatched
   */
  CDavRequest* m_pActiveRequest;
 private:
#ifdef HAVE_LIBZ
  bool m_bUseCompression;
#endif /*HAVE_LIBZ*/
  bool m_bExpect100;
  bool m_bHttp11;
  bool m_bPersistent;
  bool m_bUseSSL;
  bool m_bUseProxy;
  CDavSocket* m_pSocket;
  CDavAuthManager* m_pAuthManager;
  CDavXmlBuffer* m_pXmlBuffer;
  CDavMemBuffer* m_pMemBuffer;
  CDavServerPolicy* m_pServerPolicy;
  CDavSSLCertificateList* m_pUnauthenticCerts;
  CDavLock* m_pCurrentLock;	//pass info to/from callback in lock methods
  OI_CONN_STATE m_enuConnState;
};

OI_END_CLASSEXPORT

#endif /*ONION_HEADER_DAVWORKSESSION_H*/
