/* 
   Container of XML nodes
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavXmlBuffer.cpp 506 2016-09-27 07:49:57Z tfbuilder $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/DavXmlBuffer.h>
#include <onion/DavXmlErrorHandler.h>
#include <xercesc/dom/DOM.hpp>
#include <xercesc/framework/LocalFileFormatTarget.hpp>
#include <xercesc/parsers/XercesDOMParser.hpp>
#include <xercesc/framework/MemBufInputSource.hpp>

CDavXmlBuffer::CDavXmlBuffer(const char* pszDocName)
{
  OI_ASSERT(pszDocName);
  m_pDoc = NULL;
  m_strDocName = pszDocName;
  TrimLeftA(m_strDocName);
  TrimRightA(m_strDocName);
  if(m_strDocName.length() == 0)
    m_strDocName = OI_XMLCACHE_DOCNAME;
  initDocument();
}

CDavXmlBuffer::~CDavXmlBuffer()
{
  m_pDoc->release();
}

void 
CDavXmlBuffer::AppendChild(XNS(DOMElement)* pElement)
{
  OI_ASSERT(m_pDoc);
  if(!pElement) return;
  if(pElement->getOwnerDocument() == m_pDoc) return;

  XNS(DOMElement)* pRoot = m_pDoc->getDocumentElement();
  XNS(DOMNode)* pNewNode = m_pDoc->importNode(pElement, true);
  pRoot->appendChild(pNewNode);

  const XMLCh* pszPrefix = pElement->getPrefix();	//D
  const XMLCh* pszNS = pElement->getNamespaceURI();	//DAV:

  OI_STRING_A strPrefix = "xmlns:";
  strPrefix += X(pszPrefix);
  pRoot->setAttribute(X(strPrefix.c_str()), pszNS);
}

void
CDavXmlBuffer::initDocument()
{
  if(m_strDocName.empty())
    m_strDocName = OI_XMLCACHE_DOCNAME;

  if(m_pDoc)
    m_pDoc->release();
  XNS(DOMImplementation) *pImpl = 
    XNS(DOMImplementationRegistry)::getDOMImplementation(X("LS"));

  //creating the document object
  m_pDoc = pImpl->createDocument(X(OI_XML_DAVNAMESPACE),
				 X(m_strDocName.c_str()), NULL);

  //setting default namespace mapping
  XNS(DOMAttr)* pAttr = m_pDoc->createAttributeNS(X(OI_XML_NSURI),
						  X("xmlns:D"));
  pAttr->setValue(X("DAV:"));
}

void
CDavXmlBuffer::Clear()
{
  initDocument();
}

XNS(DOMDocument)*
CDavXmlBuffer::GetDocument()
{
  return m_pDoc;
}

bool
CDavXmlBuffer::ImportDocument(XNS(DOMDocument)* pDoc)
{
  if(!pDoc || !m_pDoc)
    return false;
  if(pDoc == m_pDoc)
    return true;
  XNS(DOMElement)* pExternRoot = pDoc->getDocumentElement();
  if(!pExternRoot)
    return false;
  XNS(DOMElement)* pInternalRoot = m_pDoc->getDocumentElement();
  if(pInternalRoot){
    m_pDoc->removeChild(pInternalRoot);
    pInternalRoot->release();
  }
  pInternalRoot = (XNS(DOMElement)*)(m_pDoc->importNode(pExternRoot, true));
  m_pDoc->appendChild(pInternalRoot);
  return true;

}

XNS(DOMNode)*
CDavXmlBuffer::ImportNode(XNS(DOMNode)*pNode)
{
  OI_ASSERT(pNode);
  XNS(DOMDocument)* pDoc = pNode->getOwnerDocument();
  if(!pDoc)
    return NULL;
  if(pDoc == m_pDoc)
    return pNode;
  XNS(DOMNode)* pImported = m_pDoc->importNode(pNode, true);
  return pImported;
}

bool
CDavXmlBuffer::SaveAs(const char* pszPath)
{
  bool bRet = false;
//  XNS(DOMWriter)* pWriter;
  XNS(DOMLSSerializer)* lpSSerializer;
  XNS(LocalFileFormatTarget)* pFormatTarget;
  XNS(DOMLSOutput)* lpOutput;
  XNS(DOMImplementation) *pImpl = 
    XNS(DOMImplementationRegistry)::getDOMImplementation(X("LS"));
  lpSSerializer = ((XNS(DOMImplementationLS)*)pImpl)->createLSSerializer();
  lpOutput = ((XNS(DOMImplementationLS)*)(pImpl))->createLSOutput();
  pFormatTarget = new XNS(LocalFileFormatTarget)(pszPath);
  lpOutput->setByteStream(pFormatTarget);
  lpSSerializer->setNewLine(X("\r\n"));
  bRet = lpSSerializer->write(m_pDoc, lpOutput);
  lpOutput->release();
  lpSSerializer->release();
  delete pFormatTarget;
  return bRet;
}

bool
CDavXmlBuffer::Load(const char* pszPath)
{
  bool bRet = true;
  if(!pszPath)
    return false;
  XNS(XercesDOMParser)* pParser = new XNS(XercesDOMParser);
  XNS(ErrorHandler)* pErrHandler = new CDavXmlErrorHandler;
  pParser->setDoNamespaces(true);
  pParser->setIncludeIgnorableWhitespace(false);
  pParser->setExitOnFirstFatalError(true);
  pParser->setErrorHandler(pErrHandler);
  try{
    pParser->parse(pszPath);
  } catch(XNS(DOMException) e) { 
    //DOMException
    OI_DEBUG("DOMException:%s\n", (const char*)X(e.msg));
    bRet = false;
  } catch(XNS(SAXException) e) { 
    //SAXException
    OI_DEBUG("SAXException:%s\n", (const char*)X(e.getMessage()));
    bRet = false;
  } catch(...) {
    //unknown exceptions
    OI_DEBUG("unhandled exception\n");
    bRet = false;
  }
  if(bRet)
    bRet = ImportDocument(pParser->getDocument());

  delete pParser;
  delete pErrHandler;
  return bRet;
}

/**
 * Load xml from a string buffer
 *  nLen is BYTE lenght of the buffer
 *  bAdopt indicates whether to make a copy of the buffer
 */
bool
CDavXmlBuffer::Load(const char* pszContent,
		    unsigned int unLen,
		    bool bAdopt)
{
  bool bRet = true;
  if(!pszContent || !unLen)
    return false;
  XNS(MemBufInputSource)* pSource = 
    new XNS(MemBufInputSource)((const XMLByte*)pszContent,
			       unLen,
			       m_strDocName.c_str(),
			       bAdopt);

  XNS(XercesDOMParser)* pParser = new XNS(XercesDOMParser);
  XNS(ErrorHandler)* pErrHandler = new CDavXmlErrorHandler;
  pParser->setDoNamespaces(true);
  pParser->setIncludeIgnorableWhitespace(false);
  pParser->setExitOnFirstFatalError(true);
  pParser->setErrorHandler(pErrHandler);
  try{
    pParser->parse(*pSource);
  } catch(XNS(DOMException) e) { 
    //DOMException
    OI_DEBUG("DOMException:%s\n", (const char*)X(e.msg));
    bRet = false;
  } catch(XNS(SAXException) e) { 
    //SAXException
    OI_DEBUG("SAXException:%s\n", (const char*)X(e.getMessage()));
    bRet = false;
  } catch(...) {
    //unknown exceptions
    OI_DEBUG("unhandled exception\n");
    bRet = false;
  }
  if(bRet)
    bRet = ImportDocument(pParser->getDocument());

  delete pParser;
  delete pErrHandler;
  delete pSource;
  return bRet;
}

CDavXmlBuffer&
CDavXmlBuffer::operator=(CDavXmlBuffer& master)
{
  XNS(DOMDocument)* pDoc = master.GetDocument();
  ImportDocument(pDoc);
  return *this;
}
