/* 
   Types of onion library
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: OnionTypes.h 558 2017-10-29 01:11:56Z tfbuilder $
*/

#ifndef ONION_HEADER_ONIONTYPES_H
#define ONION_HEADER_ONIONTYPES_H 1
#include <string>
#include <sstream>

enum OI_RESULT{
  //generic errors
  OIGEUNKNOWN		= -1,	//unknown generic error
  OIGENOTIMPLEMENTED	= -2,	//feature not implemented
  OIGEINVALIDARG	= -3,	//invalid argument
  OIGEINVALIDSTATE	= -4,	//invalid state
  OIGEFILEOPENFAILED	= -5,	//failed to open file

  //status code
  OI_OK			= 0,
  OI_RETRY		= 1,
  OI_USERCANCELED	= 2,

  //encryption(ssl) errors
  OIEEGENERIC		= 201,	//Generic SSL Error
  OIEENOENTROPY		= 202,	//No entropy source found. could not seed PRNG
  OIEELACKOFENTROPY	= 203,	//SSL disabled due to lack of entropy
  OIEEDATAINITFAILED	= 204,	//SSL data initialization failed
  OIEENOSERVERCERT	= 206,	//SSL server certificate not present
  OIEEINVALIDCERT	= 207,	//SSL server certificate invalid
  OIEECERTCHANGED	= 208,	//Server certificate changed, connection may be intercepted
  OIEECERTCHECKFAILED	= 209,	//Certificate check failed
  OIEECNMISMATCH	= 210,	//CN Mismatch
  OIEETUNNELFAILED	= 211,	//proxy tunnel failed
  OIEEHANDSHAKEFAILED	= 212,	//SSL handshake failed
  OIEEFATALCONNERROR	= 213,	//fatal error encountred while trying to connect via SSL

  //HTTP protocol errors
  OIHELINETOOLONG	= 301,	//HeaderLine Too Long
  OIHETOOMANYHEADERS	= 302,	//Too many header fields
  OIHEPROTOCOLPANIC	= 303,	//The response in not a Http status
  OIHECHUNKSIZE		= 304,	//response chunk size parse error
  OIHEMALFORMEDHEADER	= 305,	//malformed http header

  //XML errors
  OIXEINITFAILED	= 401,	//Xerces init failed
  OIXEPARSERCREATION	= 402,	//Parser creation failed
  OIXEEXCEPTION		= 403,	//XML exception

  //socket errors
  OISEGEN		= 501,	//generic(unclear) error
  OISETIMEOUT		= 502,	//Connection timed out
  OISECLOSED		= 503,	//Socket Closed
  OISERESET		= 504,	//Connection reset by server
  OISELINETOOLONG	= 505,	//Line Too Long
  OISECONNABORTED	= 506,	//Connection aborted by server

  OISEINITFAILED	= 507,	//socket version too low
  OISEINITIALIZED	= 508,	//socket already initialized
  OISECLEANUPFAILED	= 509,	//socket clean up failed
  OISEHOSTNOTFOUND	= 510,	//host not found
  OISEINVALIDSOCK	= 511,	//can not get sock from system
  OISESOCKVER		= 512,	//socket version not supported by OS
  OISECONNFAILED	= 513,	//system connection failed
  OISETRUNC		= 514,	//connection truncated
  OISEREADFAILED	= 515,	//unable to read from socket

  //resource parse errors
  OIRESYNTAX		= 601,	//XML syntax error
  OIREHREFNOTFOUND	= 602,	//href element not found while parsing response

  //authentication errors
  OIAEHEADERSYNTAX	= 701,	//header is not well-formed
  OIAEFAILEDTOGETCRED	= 702,	//failed to get credential(the OnAuthentication handler returned false)
  OIAEDIGMISMATCH	= 703,	//Digest mismatch, connection could be intercepted
  OIAEINVALIDSCHEME	= 704,	//invalid auth scheme
  OIAECNONCEMISMATCH	= 705,	//cnonce mismatch
  OIAENONCEMISMATCH	= 706,	//nonce mismatch

  //DAV errors
  OIDENOCONTENT		= 801,	//204 no content
  OIDEMULTISTATUS	= 802,	//207 multistatus
  OIDEBADREQUEST	= 803,	//400 Bad Request
  OIDEFORBIDDEN		= 804,	//403 forbidden
  OIDENOTFOUND		= 805,	//404 not found
  OIDENOTALLOWED	= 806,	//405 not allowed
  OIDECONFLICT		= 807,	//409 conflict
  OIDEPCONFAILED	= 808,	//412 Precondition Failed
  OIDELOCKED		= 809,	//423 Locked
  OIDEFAILEDDEPS	= 810,	//424 Failed Dependency
  OIDEINTRSVRERR	= 812,	//500 Internal Server Error
  OIDEBADGATEWAY	= 813,	//502 Bad Gateway
  OIDESERVICENOTAVAIL	= 814,	//503 Service Not Available
  OIDECTYPEMISMATCH	= 818,	//unexpected content type
  OIDEMOVEDPERMANENTLY	= 819,	//301 Moved Permanently
  OIDEMOVEDTEMPORARILY	= 820,	//302 Moved Temporarily
  OIDEINSUFFSTG		= 821,	//507 Insufficient Storage
  OIDEPAYMENTREQUIRED	= 822,	//402 Payment Required
  OIDEENTITYTOOLARGE = 823,	// 312 Entity to Large
  OIDEMISDIRECTEDREQUEST = 824, // 421 Misdirected Request

  //system errors
  OIYEREQNOTFOUND	= 901,	//request not found in library's config
  OIYESTATUSNOTFOUND	= 902,	//status code not found in library config
  OIYECTYPENOTFOUND	= 903,	//content type not found library config
  OIYECONFINITFAILED	= 904,	//server profile config initialization failed	
  OIYECONFSYNTAX	= 905,	//server profile config syntax error
  OIYEILLEGALOPCODE	= 906,	//fatal error: illegal OP code
  OIYEOPNOTSUPPORTED	= 907,	//this operation is not supported

#ifdef HAVE_LIBZ
  //zlib errors
  OIZEINITFAILED	= 1001,	//zlib initializatiion failed
  OIZEBADHEADER		= 1002,	//bad gzip header
  OIZEBADDATA		= 1003	//bad gzip data
#endif /*HAVE_LIBZ*/
};

enum OI_OPCODE {
  OP_NONE		= -1,
  OP_USE_DEFAULT	= 0,	//use default body consumer
  OP_SIMPLE_ERR		= 1,	//return simple error code

  OP_PARSE_ERR_XML	= 3,	//construct an xml parser and parse xml error
  OP_CACHE_HTML		= 4,	//put response body in session cache

  OP_AUTHENTICATE	= 7	//use auth-manager to handle authentication header
};

enum OI_REQ_TYPE {
  T_REQ_INVALID		= 0,
  T_REQ_CONNECT		= 1,
  T_REQ_COPY		= 2,
  T_REQ_DELETE		= 3,
  T_REQ_GET		= 4,
  T_REQ_LOCK		= 5,
  T_REQ_MKCOL		= 6,
  T_REQ_MOVE		= 7,
  T_REQ_OPTIONS		= 8,
  T_REQ_POST		= 9,
  T_REQ_PROPFIND	= 10,
  T_REQ_PROPPATCH	= 11,
  T_REQ_PUT		= 12,
  T_REQ_UNLOCK		= 13,
  T_REQ_SEARCH		= 14,
  T_REQ_HEAD		= 15,
  T_REQ_CUSTOM		= 255
};

enum OI_RSP_MODE {
  M_RSP_TILLEOF		= 0,	// read till eof 
  M_RSP_NO_BODY		= 1,	// implicitly no body (method HEAD, 204 No Content, 304)
  M_RSP_CHUNKED		= 2,	// using chunked transfer-encoding 
  M_RSP_CLENGTH		= 3	// using given content-length 
};

enum OI_CONN_STATE {
  S_CONN_DISCONNECTED,
  S_CONN_RESOLVE,		// resolving host name
  S_CONN_CONNECTING,		// connecting
  S_CONN_CONNECTED,		// connection established
  S_CONN_TUNNELED		// proxy tunnel established
};


enum OI_RBP_TYPE {
  T_RBP_UNKNOWN		= -1,
  T_RBP_DUMMY		= 0,
  T_RBP_BUFFER		= 1,
  T_RBP_CUSTOM		= 2,
  T_RBP_FILE		= 3,
  T_RBP_XML		= 4
};

enum OI_PBC_TYPE {
  T_PBC_UNKNOWN		= -1,
  T_PBC_DUMMY		= 0,
  T_PBC_BUFFER		= 1,
  T_PBC_FILE		= 3,
  T_PBC_XML		= 4
};

enum OI_AUTH_CLASS {
  C_AUTH_INVALID	= 0,
  C_AUTH_SERVER		= 1,
  C_AUTH_PROXY		= 2
};

/*TODO: removed the "invalid" scheme*/
enum OI_AUTH_SCHEME {
  H_AUTH_INVALID	= 0,
  H_AUTH_NOAUTH		= 1,
  H_AUTH_BASIC		= 2,
  H_AUTH_DIGEST		= 3,
  H_AUTH_NTLM		= 4
};

enum OI_AUTH_STATE {
  S_AUTH_UNAUTHENTICATED= 1,
  S_AUTH_AUTHENTICATED	= 2,
  S_AUTH_NTLMTYPE1	= 3,
  S_AUTH_NTLMTYPE2	= 4,
  S_AUTH_NTLMTYPE3	= 5
};

enum OI_AUTH_QOP {
  Q_AUTH_NONE		= 1,
  Q_AUTH_AUTH		= 2,
  Q_AUTH_AUTHINT	= 3
};

//resource node type
enum OI_RES_TYPE {
  T_RES_NORMAL		= 1,
  T_RES_COLLECTION	= 2
};

enum OI_LOCK_TYPE {
  T_LOCK_WRITE		= 1
};

enum OI_LOCK_SCOPE {
  SCP_LOCK_EXCLUSIVE	= 1,
  SCP_LOCK_SHARED	= 2
};

enum OI_LOCK_DEPTH {
  D_LOCK_ZERO		= 1,
  D_LOCK_INFINITE	= 2
};

enum OI_PFIND_DEPTH {
  D_PFIND_ZERO		= 1,
  D_PFIND_ONE		= 2,
  D_PFIND_INFINITE	= 3
};

enum OI_SSL_PKEYALG {
  ALG_PKEY_NONE,
  ALG_PKEY_RSA,
  /*ALG_PKEY_RSA2,*/
  ALG_PKEY_DSA,
  /*ALG_PKEY_DSA1,
  ALG_PKEY_DSA2,
  ALG_PKEY_DSA3,
  ALG_PKEY_DSA4,*/
  ALG_PKEY_DH
};

enum OI_SSL_CRTMSG {
  MSG_CRT_X509,
  MSG_CRT_PKCS7
};

enum OI_SSL_CRTENC {
  ENC_CRT_DER,
  ENC_CRT_PEM
};

enum OI_STREAM_MODE {
  M_STM_READ,
  M_STM_WRITE
};

typedef std::basic_string<char>		OI_STRING_A;
typedef std::stringstream		OI_STRSTREAM_A;

typedef unsigned char			OI_BYTE;

/* largefile support */
#ifdef WIN32
   typedef unsigned __int64		OI_SIZE_T;
   typedef __int64			OI_SSIZE_T;
#  define OI_STRTOUL(p, pp, l)		_atoi64(p)
#  define OI_FSTAT(h, p)		_fstati64(h, p)
#  define OI_SIZEFORMAT			"%I64u"
#  define OI_SSIZEFORMAT		"%I64"
#elif defined (__MWERKS__) /*code warrior TODO: on mac osx???*/
   typedef unsigned long		OI_SIZE_T;
   typedef long				OI_SSIZE_T;
#  define OI_STRTOUL(p, pp, l)          atoi(p)
#  define OI_FSTAT(h, p)                fstat(h, p)
#  define OI_SIZEFORMAT                 "%lu"
#  define OI_SSIZEFORMAT		"%ld"
#else /* POSIX platforms */
   typedef off_t			OI_SIZE_T;
   typedef off_t			OI_SSIZE_T;
#  if defined(_FILE_OFFSET_BITS) && (_FILE_OFFSET_BITS == 64) /*Linux,Solaris*/
#    define OI_USESTAT64
#    define OI_STRTOUL(p, pp, l)	strtoull(p, pp, l)
#    define OI_FSTAT(h, p)		fstat64(h, p)
#    define OI_SIZEFORMAT		"%llu"
#    define OI_SSIZEFORMAT		"%lld"
#  elif SIZEOF_OFF_T >= 8 /*BSD platforms with large file*/
#    define OI_STRTOUL(p, pp, l)	strtoul(p, pp, l)
#    define OI_FSTAT(h, p)		fstat(h, p)
#    define OI_SIZEFORMAT		"%llu"
#    define OI_SSIZEFORMAT		"%lld"
#  else
#    define OI_STRTOUL(p, pp, l)	strtoul(p, pp, l)
#    define OI_FSTAT(h, p)		fstat(h, p)
#    define OI_SIZEFORMAT		"%lu"
#    define OI_SSIZEFORMAT		"%ld"
#  endif
#endif

typedef struct tagOIOPERATION {
  OI_OPCODE enuOpCode;
  OI_RESULT enuErrCode;
  char szCutElement[OI_MAX_ELEM_LEN + 1];
} OIOPERATION;

typedef struct tagOIPKEYINFO {
  OI_SSL_PKEYALG enuAlgorithm;
  int nKeyLength;	/* only valid for RSA keys */
} OIPKEYINFO;

typedef struct tagOICRANGE {
  OI_SSIZE_T nStart;
  OI_SSIZE_T nEnd;
  OI_SSIZE_T nTotal;
} OICRANGE;

#endif /*ONION_HEADER_ONIONTYPES_H*/
