/* 
   Request body provider that provides XML contents
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: RBPVDXml.cpp 506 2016-09-27 07:49:57Z tfbuilder $
*/


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/RBPVDXml.h>
#include <onion/DavXmlErrorHandler.h>
#include <xercesc/parsers/XercesDOMParser.hpp>


CRBPVDXml::CRBPVDXml(const char* pszDocName):
  m_xbfBody(pszDocName)
{
  OI_ASSERT(pszDocName);
  m_pszBuf = NULL;
  m_unBufLen = 0;
  m_unCurPos = 0;

  XNS(DOMImplementation) *pImpl = 
    XNS(DOMImplementationRegistry)::getDOMImplementation(X("LS"));

  //creating document
  //set default namespace mapping
  //seems unnecessary in Xercesc 2.3.0
  /*
    DOMAttr* pAttr = m_pDoc->createAttributeNS(X(OI_XML_NSURI), X("xmlns:D"));
    pAttr->setValue(X("DAV:"));
    DOMElement* pRoot =  m_pDoc->getDocumentElement();
    pRoot->setAttributeNode(pAttr);
  */

  //creating the writer
  m_lpDomSSerializer = ((XNS(DOMImplementationLS)*)pImpl)->createLSSerializer();
  m_lpOutput = ((XNS(DOMImplementationLS)*)(pImpl))->createLSOutput();
  m_pFormatTarget = new XNS(MemBufFormatTarget);

}

CRBPVDXml::~CRBPVDXml()
{
	m_lpOutput->release();
	m_lpDomSSerializer->release();
	//if(m_pFormatTarget) 
  delete m_pFormatTarget;
}

OI_RBP_TYPE
CRBPVDXml::GetType()
{
  return T_RBP_XML;
}

void
CRBPVDXml::OnCancel(CDavRequest* pReq)
{
}

void
CRBPVDXml::Close()
{
  m_pFormatTarget->reset();
}

OI_SIZE_T
CRBPVDXml::GetBodyLength()
{
  return m_unBufLen;
}

OI_RESULT
CRBPVDXml::Rewind()
{
  m_unCurPos = 0;
  return OI_OK;
}

void
CRBPVDXml::GenerateXml()
{
	if (!(m_lpDomSSerializer && m_pFormatTarget))
    return;

  XNS(DOMDocument)* pDoc =  m_xbfBody.GetDocument();
  OI_ASSERT(pDoc);
  XNS(DOMNode)* pRoot = (XNS(DOMNode)*)pDoc;
  m_lpDomSSerializer->getDomConfig()->setParameter(XNS(XMLUni)::fgDOMWRTSplitCdataSections, false);
  m_lpOutput->setByteStream(m_pFormatTarget);
  m_lpDomSSerializer->write(pDoc, m_lpOutput);
  m_pszBuf = (char*)(m_pFormatTarget->getRawBuffer());

  m_unBufLen = m_pFormatTarget->getLen();
  m_unCurPos = 0;
}

OI_RESULT
CRBPVDXml::Open()
{
  GenerateXml();
  return OI_OK;
}

OI_RESULT
CRBPVDXml::Read(CDavRequest* pReq, char* pszBuf, size_t* punReadLen)
{
  if(m_unBufLen == 0 || (m_unCurPos >= m_unBufLen)){
    *punReadLen = 0;
    return OI_OK;
  }

  size_t unReadLen = m_unBufLen - m_unCurPos;

  if(unReadLen > *punReadLen) 
    unReadLen = *punReadLen;
  memcpy(pszBuf, m_pszBuf + m_unCurPos, unReadLen);
  //TODO: memcpy is not always available
  m_unCurPos += unReadLen;
  *punReadLen = unReadLen;

  return OI_RETRY;
}


void
CRBPVDXml::AddRootElement(const char* pszNS, const char* pszName)
{
  OI_ASSERT(pszNS);
  OI_ASSERT(pszName);

  XNS(DOMDocument)* pDoc =  m_xbfBody.GetDocument();
  OI_ASSERT(pDoc);

  XNS(DOMElement)* pRoot = pDoc->getDocumentElement();
  XNS(DOMElement)* pProp = pDoc->createElementNS(X(pszNS),X(pszName));
  pRoot->appendChild(pProp);
}


void
CRBPVDXml::AddNamespaceMapping(const char* pszPrefix, const char* pszURI)
{
  OI_ASSERT(pszPrefix);
  OI_ASSERT(pszURI);


  XNS(DOMDocument)* pDoc =  m_xbfBody.GetDocument();
  OI_ASSERT(pDoc);

  XNS(DOMElement)* pRoot =  pDoc->getDocumentElement();
  if(!pRoot)
    return;
  pRoot->setAttribute(X(pszPrefix), X(pszURI));
}


void
CRBPVDXml::SetDocumentName(const char* pszName)
{
  OI_ASSERT(pszName);

  XNS(DOMDocument)* pDoc =  m_xbfBody.GetDocument();
  OI_ASSERT(pDoc);

  XNS(DOMElement)* pRoot =  pDoc->getDocumentElement();
  pDoc->renameNode(pRoot, X(OI_XML_DAVNAMESPACE), X(pszName));
}


bool
CRBPVDXml::Load(const char* pszPath)
{
  if(!pszPath)
    return false;

  bool bRet = m_xbfBody.Load(pszPath);
  return bRet;
}


bool
CRBPVDXml::Load(CDavXmlBuffer* pXmlBuffer)
{
  if(!pXmlBuffer)
    return false;
  m_xbfBody = *pXmlBuffer;
  return true;
}

