/* 
   Utilty functions
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 1999-2003, Joe Orton <joe@manyfish.co.uk>
   Copyright (C) 1999-2004, Joe Orton <joe@manyfish.co.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: Utils.cpp 565 2017-10-29 02:15:38Z yone $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <ctype.h>
#include <onion/Utils.h>
#include <onion/DavSSLCertificateList.h>
#include <onion/DavSSLClientCertificate.h>
#include <onion/DavSocket.h>
#include <onion/DavWorkSession.h>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/util/XMLUTF8Transcoder.hpp>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/pkcs12.h> /* for PKCS12_PBE_add */
#include <openssl/rand.h>
#include <openssl/md5.h>
#include <openssl/bn.h>
#include <time.h>
#include <memory>

#define OI_ASC2HEX(x) (((x) <= '9') ? ((x) - '0') : (tolower((x)) + 10 - 'a'))
#define OI_HEX2ASC(x) ((x) > 9 ? ((x) - 10 + 'a') : ((x) + '0'))

//private funtions in DavSocket.cpp
extern OI_RESULT Onion_initSocket();
extern void Onion_cleanupSocket();
		
//Global variables
SSL_CTX* g_pSSLCtx = NULL;
XNS(XMLUTF8Transcoder)*	g_pUTF8Transcoder = NULL;
ONIONDEBUGLOCKCALLBACK	g_pfnDebugLockCallback = NULL;
FILE* g_pDebugStream = NULL;
unsigned int g_unDebugOptions = 0;

//callbacks

static int
fnProvideClientCertificate(SSL* pSSL,
			   X509**  ppX509,
			   EVP_PKEY** ppKey)
{
  if(!pSSL || !ppX509 || !ppKey)
    return 0;
  int nRet = 0;
  CDavSocket* pSocket = (CDavSocket*)SSL_get_app_data(pSSL);
  CDavWorkSession* pSession = pSocket->GetSession();
  STACK_OF(X509_NAME)* pCAList = SSL_get_client_CA_list(pSSL);
  if(!pCAList)
    return 0;
  int nCount = sk_X509_NAME_num(pCAList);
  CDavSSLDName* pDNames = NULL;

  /* construct our CA list */
  if(nCount) {
    pDNames = new CDavSSLDName[nCount];
    int nIndex;
    for(nIndex = 0; nIndex < nCount; nIndex++){
      X509_NAME* pName = sk_X509_NAME_value(pCAList, nIndex);
      if(pName)
	pDNames[nIndex].ParseDName(pName);
    }
  }

  /* call the handler */
  CDavSSLClientCertificate clientCert;
  if(pSession->OnProvideClientCertificate(&clientCert, pDNames, nCount)) {
    if(clientCert.IsValid()) {
      X509* pX509 = clientCert.Cert().GetX509();
      CRYPTO_add(&pX509->references, 1, CRYPTO_LOCK_X509);
      EVP_PKEY* pKey = clientCert.Key();
      CRYPTO_add(&pKey->references, 1, CRYPTO_LOCK_EVP_PKEY);
      *ppX509 = pX509;
      *ppKey = pKey;
      nRet = 1;
    }
  }
  
  /* cleanup */
  if(pDNames)
    delete[] pDNames;

  return nRet;
}


//Global functions
int 
OnionNumOfLocks()
{
  return CRYPTO_num_locks();
}

void
OnionSetDebugLockCallback(ONIONDEBUGLOCKCALLBACK pfnDebugLockCallback)
{
  g_pfnDebugLockCallback = pfnDebugLockCallback;
}

OI_RESULT
OnionRedirectDebugOutput(const char* pszPath, const char* pszMode)
{
  if(g_pDebugStream){
    fclose(g_pDebugStream);
    g_pDebugStream = NULL;
  }
  if(pszPath && pszMode){
    g_pDebugStream = freopen(pszPath, pszMode, stderr);
    if(!g_pDebugStream)
      return OIGEFILEOPENFAILED;
  }
  return OI_OK;
}

void
OnionDebugOutput(const char* pszFormat, ...)
{
  OI_ASSERT(pszFormat);
  if(g_unDebugOptions & OI_DEBUG_SILENT)
    return;
  va_list params;
  va_start(params, pszFormat);
  vfprintf(stderr, pszFormat, params);
  va_end(params);
  if(g_unDebugOptions & OI_DEBUG_FLUSH)
    fflush(stderr);
}

void
OnionSetDebugOptions(unsigned int unOptions)
{
  g_unDebugOptions = unOptions;
}

OI_RESULT 
OnionInitialize(OSSLLOCKCALLBACK sslLockCallback)
{
  OI_RESULT enuRet = Onion_initSocket();
  if(OI_OK != enuRet)
    return enuRet;

  XNS(XMLPlatformUtils)::Initialize();
  if(!g_pUTF8Transcoder){
    XNS(XMLTransService)::Codes resValue;
    g_pUTF8Transcoder = (XNS(XMLUTF8Transcoder)*)XNS(XMLPlatformUtils)::
      fgTransService->makeNewTranscoderFor(XNS(XMLRecognizer)::UTF_8,
					   resValue,
					   16384);
    if(!g_pUTF8Transcoder)
      return OIXEINITFAILED;
  }

  if(sslLockCallback)
    CRYPTO_set_locking_callback(sslLockCallback);
  SSL_load_error_strings();
  SSL_library_init();
  PKCS12_PBE_add();  // !!! not sure why this is needed.
  if(!g_pSSLCtx) {
    g_pSSLCtx = SSL_CTX_new(SSLv23_client_method());
    SSL_CTX_set_client_cert_cb(g_pSSLCtx, fnProvideClientCertificate);
  }

  if (RAND_status() == 1)	
    return OI_OK; // already seeded internally by OpenSSL

  //***** NOTICE: RAND_status() always returns 1 
  //***** so the following code is neve executed
  char* pszFile = getenv("EGDSOCKET");
  if (pszFile && RAND_egd(pszFile) != -1) 
    return OI_OK;

  pszFile = getenv("HOME");
  if (pszFile){
    OI_STRING_A strPath;
    strPath = pszFile;
    strPath += "/.entropy";
    /*
      char szPath[1024];    
      sprintf(szPath,  "%s/.entropy", pszFile);
      if (RAND_egd(szPath) != -1)
        return OI_OK;
    */
    if (RAND_egd(strPath.c_str()) != -1)
      return OI_OK;
  }
  return OIEENOENTROPY;
}

void 
OnionTerminate()
{
  if(g_pDebugStream){
    fclose(g_pDebugStream);
    g_pDebugStream = NULL;
  }
  if(g_pSSLCtx){
    SSL_CTX_free(g_pSSLCtx);
    g_pSSLCtx = NULL;
  }
  CRYPTO_set_locking_callback(NULL);
  /*	in case OpenSSL does not cleanup automatically
	EVP_cleanup();
	ENGINE_cleanup();
	CRYPTO_cleanup_all_ex_data();
	ERR_remove_state(0);
	ERR_free_strings();
  */

  if(g_pUTF8Transcoder)
    delete g_pUTF8Transcoder;
  XNS(XMLPlatformUtils)::Terminate();

  Onion_cleanupSocket();
}

const char*	
GetErrorDescription(OI_RESULT enuResult)
{
  switch(enuResult){
  case OIGEUNKNOWN:
    return "unknown generic error";
  case OIGENOTIMPLEMENTED:
    return "feature not implemented";
  case OIGEINVALIDARG:
    return "invalid argument";
  case OIGEINVALIDSTATE:
    return "invalid state";
  case OIGEFILEOPENFAILED:
    return "failed to open file";

    //status code
  case OI_OK:
    return "No Error";
  case OI_RETRY:
    return "Retry";
  case OI_USERCANCELED:
    return "User canceled operation";

    //encryption(ssl) errors
  case OIEEGENERIC:
    return "Generic SSL Error";
  case OIEENOENTROPY:
    return "No entropy source found. could not seed PRNG";
  case OIEELACKOFENTROPY:
    return "SSL disabled due to lack of entropy";
  case OIEEDATAINITFAILED:
    return "SSL data initialization failed";
  case OIEENOSERVERCERT:
    return "SSL server certificate not present";
  case OIEEINVALIDCERT:
    return "SSL server certificate invalid";
  case OIEECERTCHANGED:
    return "Server certificate changed, connection may be intercepted";
  case OIEECERTCHECKFAILED:
    return "Certificate check failed";
  case OIEECNMISMATCH:
    return "CN Mismatch";
  case OIEETUNNELFAILED:
    return "proxy tunnel failed";
  case OIEEHANDSHAKEFAILED:
    return "SSL handshake failed";
  case OIEEFATALCONNERROR:
    return "fatal error encountred while trying to connect via SSL";


    //HTTP protocol errors
  case OIHELINETOOLONG:
    return "HeaderLine Too Long";
  case OIHETOOMANYHEADERS:
    return "Too many header fields";
  case OIHEPROTOCOLPANIC:
    return "The response in not a Http status";
  case OIHECHUNKSIZE:
    return "response chunk size parse error";
  case OIHEMALFORMEDHEADER:
    return "malformed http header";


    //XML errors
  case OIXEINITFAILED:
    return "Xerces init failed";
  case OIXEPARSERCREATION:
    return "Parser creation failed";
  case OIXEEXCEPTION:
    return "XML exception";

    //socket errors
  case OISEGEN:
    return "generic(unclear) error";
  case OISETIMEOUT:
    return "Connection timed out";
  case OISECLOSED:
    return "Socket Closed";
  case OISERESET:
    return "Connection reset by server";
  case OISELINETOOLONG:
    return "http status line Too Long";
  case OISECONNABORTED:
    return "Connection aborted by server";

  case OISEINITFAILED:
    return "socket version too low";
  case OISEINITIALIZED:
    return "socket already initialized";
  case OISECLEANUPFAILED:
    return "socket clean up failed";
  case OISEHOSTNOTFOUND:
    return "host not found";
  case OISEINVALIDSOCK:
    return "can not get sock from system";
  case OISESOCKVER:
    return "socket version not supported by OS";
  case OISECONNFAILED:
    return "system connection failed";
  case OISETRUNC:
    return "connection truncated";
  case OISEREADFAILED:
    return "unable to read from socket";

    //resource parse errors
  case OIRESYNTAX:
    return "XML syntax error";
  case OIREHREFNOTFOUND:
    return "href element not found while parsing response";

    //authentication errors
  case OIAEHEADERSYNTAX:
    return "header is not well-formed";
  case OIAEFAILEDTOGETCRED:
    return "failed to get credential(the OnAuthentication handler returned false)";
  case OIAEDIGMISMATCH:
    return "Digest mismatch, connection could be intercepted";
  case  OIAEINVALIDSCHEME:
    return "invalid auth scheme";
  case OIAECNONCEMISMATCH:
    return "cnonce mismatch";
  case OIAENONCEMISMATCH:
    return "nonce mismatch";


    //DAV errors
  case OIDENOCONTENT:
    return "204 no content";
  case OIDEMULTISTATUS:
    return "207 multistatus";
  case OIDEBADREQUEST:
    return "400 Bad Request";
  case OIDEFORBIDDEN:
    return "403 forbidden";
  case OIDENOTFOUND:
    return "404 not found";
  case OIDENOTALLOWED:
    return "405 not allowed";
  case OIDECONFLICT:
    return "409 conflict";
  case OIDEPCONFAILED:
    return "412 Precondition Failed";
  case OIDELOCKED:
    return "423 Locked";
  case OIDEFAILEDDEPS:
    return "424 Failed Dependency";
  case OIDEINTRSVRERR:
    return "500 Internal Server Error";
  case OIDEBADGATEWAY:
    return "502 Bad Gateway";
  case OIDESERVICENOTAVAIL:
    return "503 Service Not Available";
  case OIDECTYPEMISMATCH:
    return "unexpected content type";
  case OIDEMOVEDPERMANENTLY:
    return "301 Moved Permanently";
  case OIDEMOVEDTEMPORARILY:
    return "302 Moved Temporarily";
  case OIDEINSUFFSTG:
    return "507 Insufficient Storage";
  case OIDEPAYMENTREQUIRED:
    return "402 Payment Required";
  case OIDEENTITYTOOLARGE:
    return "314 Entity too large";
  case OIDEMISDIRECTEDREQUEST:
    return "421 Misdirected Request";

    //system errors
  case OIYEREQNOTFOUND:
    return "request not found in library's config";
  case OIYESTATUSNOTFOUND:
    return "status code not found in library config";
  case OIYECTYPENOTFOUND:
    return "content type not found in library config";
  case OIYECONFINITFAILED:
    return "server profile config initialization failed	";
  case OIYECONFSYNTAX:
    return "server profile config syntax error";
  case OIYEILLEGALOPCODE:
    return "fatal error: illegal OP code";
  case OIYEOPNOTSUPPORTED:
    return "this operation is not supported";

#ifdef HAVE_LIBZ

    //zlib errors
  case OIZEINITFAILED:
    return "zlib initializatiion failed";
  case OIZEBADHEADER:
    return "bad gzip header";
  case OIZEBADDATA:
    return "bad gzip data";

#endif /*HAVE_LIBZ*/

  default:
    return "error not found";
  }
}

OI_STRING_A	
Unescape(const OI_STRING_A& uri)
{
  OI_STRING_A::const_iterator it = uri.begin();
  OI_STRSTREAM_A ss;
  char buf[5] = { "0x00" };

  while(it != uri.end()){
    if(*it == '%'){
      if(!isxdigit(*(it+1)) || !isxdigit(*(it+2))){
	//invalid escapped string format
	return ss.str();
      }
			
      buf[2] = *(++it);
      buf[3] = *(++it);
      ss << (char)strtol(buf, NULL, 16);
    } else {
      ss << *it;
    }
    it++;
  }
  return ss.str();
}

/* RFC2396 spake:
 * "Data must be escaped if it does not have a representation 
 * using an unreserved character".
 */

/* Lookup table: character classes from 2396. (This is overkill) */

#define SP 0   /* space    = <US-ASCII coded character 20 hexadecimal>                 */
#define CO 0   /* control  = <US-ASCII coded characters 00-1F and 7F hexadecimal>      */
#define DE 0   /* delims   = "<" | ">" | "#" | "%" | <">                               */
#define UW 0   /* unwise   = "{" | "}" | "|" | "\" | "^" | "[" | "]" | "`"             */
#define MA 1   /* mark     = "-" | "_" | "." | "!" | "~" | "*" | "'" | "(" | ")"       */
#define AN 2   /* alphanum = alpha | digit                                             */
#define RE 2   /* reserved = ";" | "/" | "?" | ":" | "@" | "&" | "=" | "+" | "$" | "," */

static const char chUriChars[128] = {
  /*                +2      +4      +6      +8     +10     +12     +14     */
  /*   0 */ CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO,
  /*  16 */ CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO, CO,
  /*  32 */ SP, MA, DE, DE, RE, DE, RE, MA, MA, MA, MA, RE, RE, MA, MA, RE,
  /*  48 */ AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, RE, RE, DE, RE, DE, RE,
  /*  64 */ RE, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN,
  /*  80 */ AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, UW, UW, UW, UW, MA,
  /*  96 */ UW, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN,
  /* 112 */ AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, AN, UW, UW, UW, MA, CO 
};

#define ESCAPE(ch) (((const signed char)(ch) < 0 || \
		    chUriChars[(unsigned int)(ch)] == 0))

#undef SP
#undef CO
#undef DE
#undef UW
#undef MA
#undef AN
#undef RE

OI_STRING_A 
Escape(const OI_STRING_A& uri) 
{
  OI_STRING_A::const_iterator it = uri.begin();
  OI_STRSTREAM_A ss;
  char buf[4] = {"???"};

  while(it != uri.end()){
    if(ESCAPE(*it)){
      sprintf(buf,"%%%02x", (unsigned char)*it);
      ss<<buf;
    } else {
      ss<<*it;
    }
    it++;
  }
  return ss.str();
}

//#undef ESCAPE

static const char chB64Alphabet[] =
"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
"abcdefghijklmnopqrstuvwxyz"
"0123456789+/=";
	
OI_STRING_A	
Base64Encode(const OI_STRING_A& input)
{
  OI_STRSTREAM_A ss;
  OI_STRING_A::const_iterator it = input.begin();

  int inlen = (int)input.length();
  for(;inlen >= 3; inlen -= 3, it += 3){
    ss << chB64Alphabet[(*it) >> 2];
    ss << chB64Alphabet[((*it) << 4 & 0x30) | (*(it+1))>>4];
    ss << chB64Alphabet[((*(it + 1)) << 2 & 0x3c) | (*(it + 2)) >> 6];
    ss << chB64Alphabet[(*(it + 2)) & 0x3f];
  }

  //processing the trailing bytes if any
  if(inlen > 0){
    ss << chB64Alphabet[(*it) >> 2];
    ss << chB64Alphabet[(((*it) << 4 & 0x30) | (inlen == 2 ? (*(it + 1)) >> 4 : 0))];
    ss << (inlen==1 ? '=' : chB64Alphabet[ (*(it + 1)) << 2 & 0x3c]);
    ss << '=';
  }
  return ss.str();
}

#define VALID_B64(ch) (((ch) >= 'A' && (ch) <= 'Z') || \
		       ((ch) >= 'a' && (ch) <= 'z') || \
		       ((ch) >= '0' && (ch) <= '9') || \
		       (ch) == '/' || (ch) == '+' || (ch) == '=')

#define DECODE_B64(ch) ((ch) >= 'a' ? ((ch) + 26 - 'a') : \
			((ch) >= 'A' ? ((ch) - 'A') : \
			((ch) >= '0' ? ((ch) + 52 - '0') : \
			((ch) == '+' ? 62 : 63))))

OI_STRING_A	
Base64Decode(const OI_STRING_A& input)
{
  OI_STRSTREAM_A ss;
  const char* empty = "";

  int inlen = (int)input.length();
  if((inlen > 0) && (inlen % 4 == 0)){
    for(OI_STRING_A::const_iterator it = input.begin();
	it != input.end();
	it += 4){
      unsigned int tmp;
      if(!VALID_B64(it[0]) || !VALID_B64(it[1]) || !VALID_B64(it[2]) ||
	 !VALID_B64(it[3]) || it[0] == '=' || it[1] == '=' ||
	 (it[2] == '=' && it[3] != '=')){
	return empty;
      }

      tmp = (DECODE_B64(it[0]) & 0x3f) << 18 |
	(DECODE_B64(it[1]) & 0x3f) << 12;
      unsigned char ch = (tmp >> 16) & 0xff;
      ss << ch;

      if (it[2] != '='){
	tmp |= (DECODE_B64(it[2]) & 0x3f) << 6;
	ch = (tmp >> 8) & 0xff;
	ss << ch;
	if (it[3] != '='){
	  tmp |= DECODE_B64(it[3]) & 0x3f;
	  ch = tmp & 0xff;
	  ss << ch;
	}
      }
    }
  }
  return ss.str();
}

#undef DECODE_B64

#undef VALID_B64

OI_STRING_A	
Utf8ToAnsi(const OI_STRING_A& sUtf8)
{
  return (const char*)X((XMLByte*)sUtf8.c_str());
}

OI_STRING_A	
AnsiToUtf8(const OI_STRING_A& sAnsi)
{
  return X(sAnsi.c_str()).UTF8();
}

void 
MakeLowerA(OI_STRING_A& strAnsi)
{
  for(OI_STRING_A::iterator itCh = strAnsi.begin(); 
      itCh != strAnsi.end(); itCh++){
    *itCh = tolower(*itCh);
  }
}

void
MakeUpperA(OI_STRING_A& strAnsi)
{
  for(OI_STRING_A::iterator itCh = strAnsi.begin(); 
      itCh != strAnsi.end(); itCh++){
    *itCh = toupper(*itCh);
  }
}

void 
TrimLeftA(OI_STRING_A& strAnsi)
{
  int nPos;
  nPos = (int)strAnsi.find_first_not_of(' ');
  if(nPos > 0)
    strAnsi.erase(0, nPos);	//trim left
}

void 
TrimRightA(OI_STRING_A& strAnsi)
{
  int nPos;
  nPos = (int)strAnsi.find_last_not_of(' ');
  if(nPos > 0)
    strAnsi.erase(nPos + 1);	//trim right
}

CDavXmlString::CDavXmlString()
{
  m_pszANSIStr = NULL;
  m_pszXMLStr = NULL;
}

CDavXmlString::CDavXmlString(const char* pszANSIStr)
{
  copy(pszANSIStr);
}

CDavXmlString::CDavXmlString(const XMLCh* pszXMLStr)
{
  copy(pszXMLStr);
}

CDavXmlString::CDavXmlString(const XMLByte* pszUTF8Str)
{
  transcode(pszUTF8Str);
}

CDavXmlString::CDavXmlString(const CDavXmlString& master)
{
  copy(master);
}

void 
CDavXmlString::copy(const char* pszANSIStr)
{
  m_pszANSIStr = XNS(XMLString)::replicate(pszANSIStr);
  m_pszXMLStr = NULL;
}

void 
CDavXmlString::copy(const XMLCh* pszXMLStr)
{
  m_pszANSIStr = NULL;
  m_pszXMLStr = XNS(XMLString)::replicate(pszXMLStr);
}

void 
CDavXmlString::copy(const CDavXmlString& master)
{
  if(master.m_pszANSIStr)
    m_pszANSIStr = XNS(XMLString)::replicate(master.m_pszANSIStr);
  else
    m_pszANSIStr = NULL;

  if(master.m_pszXMLStr)
    m_pszXMLStr = XNS(XMLString)::replicate(master.m_pszXMLStr);
  else
    m_pszXMLStr = NULL;
}

void 
CDavXmlString::transcode(const XMLByte* pszUTF8Str)
{
  int nUTF8Len = XNS(XMLString)::stringLen((const char*)pszUTF8Str) + 1;
  int nUCS2Len = nUTF8Len;
  XMLCh* pszUCS2 = new XMLCh[nUCS2Len];
  unsigned char* pSizeChars = new unsigned char[nUCS2Len];
  XMLSize_t unBytesEaten = 0;
  int nTranscodedLen =
    g_pUTF8Transcoder->transcodeFrom(pszUTF8Str,
				     nUTF8Len,
				     pszUCS2,
				     nUCS2Len,
				     unBytesEaten,
				     pSizeChars);
  delete[] pSizeChars;
  copy(pszUCS2);
  delete[] pszUCS2;
}

void 
CDavXmlString::erase()
{
  if (m_pszXMLStr) {
    XNS(XMLString)::release(&m_pszXMLStr);
    m_pszXMLStr = NULL;
  }
  if (m_pszANSIStr) {
    XNS(XMLString)::release(&m_pszANSIStr);
    m_pszANSIStr = NULL;
  }
}

CDavXmlString::~CDavXmlString()
{
  erase();
}

CDavXmlString& 
CDavXmlString::operator =(const CDavXmlString& master)
{
  erase();
  copy(master);
  return *this;
}

CDavXmlString& 
CDavXmlString::operator =(const char* pszANSIStr)
{
  erase();
  copy(pszANSIStr);
  return *this;
}

CDavXmlString& 
CDavXmlString::operator =(const XMLCh* pszXMLStr)
{
  erase();
  copy(pszXMLStr);
  return *this;
}

CDavXmlString& 
CDavXmlString::operator =(const XMLByte* pszUTF8Str)
{
  erase();
  transcode(pszUTF8Str);
  return *this;
}

unsigned int 
CDavXmlString::Length() const
{
  if(m_pszXMLStr){
    return XNS(XMLString)::stringLen(m_pszXMLStr);
  } else if(m_pszANSIStr) {
    return XNS(XMLString)::stringLen(m_pszANSIStr);
  } else {
    return 0;
  }
}

CDavXmlString& 
CDavXmlString::toEscapedUTF8()
{
  const XMLCh* pszXMLCh = operator const XMLCh*();
  int nLen = Length() + 1;
  int nUTF8Len = (nLen) * 6;
  XMLByte* pszUTF8 = new XMLByte[nUTF8Len];
  XMLSize_t unCharsEaten = 0;
  unsigned int nTranscodedLen = 
    g_pUTF8Transcoder->transcodeTo(pszXMLCh, 
				   nLen, 
				   pszUTF8, 
				   nUTF8Len,
				   unCharsEaten,
				   XNS(XMLTranscoder)::UnRep_RepChar);
  //perhaps should be UnRep_Throw

  char* pszEscaped = new char[nUTF8Len * 3];
  char buf[4] = {"???"};
  XMLByte* pchUTF8 = pszUTF8;
  char* pchEscaped = pszEscaped;
  char chUTF8;

  for( ; (chUTF8 = *pchUTF8) ; pchUTF8++){
    if(ESCAPE(chUTF8)){
      sprintf(pchEscaped,"%%%02x", (unsigned char)chUTF8);
      pchEscaped += 3;
    } else {
      *pchEscaped = chUTF8;
      pchEscaped++;
    }
  }
  *pchEscaped = '\0';

  erase();
  copy(pszEscaped);
  delete[] pszEscaped;
  delete[] pszUTF8;
  return *this;
}

/*
 * returned a utf-8 string of the content
 */
OI_STRING_A
CDavXmlString::UTF8()
{
  OI_STRING_A strRet;
  const XMLCh* pszXMLCh = operator const XMLCh*();
  int nLen = Length() + 1;
  int nUTF8Len = (nLen) * 6;
  XMLByte* pszRet = new XMLByte[nUTF8Len];
  XMLSize_t unCharsEaten = 0;
  unsigned int nTranscodedLen = 
    g_pUTF8Transcoder->transcodeTo(pszXMLCh, 
				   nLen, 
				   pszRet, 
				   nUTF8Len,
				   unCharsEaten,
				   XNS(XMLTranscoder)::UnRep_RepChar);
  //perhaps should be UnRep_Throw
  strRet = (char*)pszRet;
  delete [] pszRet;
  return strRet;
}

CDavXmlString::operator const char*()
{
  if(!m_pszANSIStr){
    if(m_pszXMLStr)
      m_pszANSIStr = XNS(XMLString)::transcode(m_pszXMLStr);
    if(!m_pszANSIStr)
      m_pszANSIStr = XNS(XMLString)::replicate("");
  }
  return m_pszANSIStr;
}

CDavXmlString::operator const XMLCh*()
{
  if(!m_pszXMLStr){
    if(m_pszANSIStr)
      m_pszXMLStr = XNS(XMLString)::transcode(m_pszANSIStr);
    if(!m_pszXMLStr)
      m_pszXMLStr = XNS(XMLString)::transcode("");
  }
  return m_pszXMLStr;
}

CDavStringTokenizer::CDavStringTokenizer(const char* pszInput, 
					 const char* pszDelimiters):
  m_pszHead(pszInput), 
  m_pszCurrentPos(pszInput),
  m_pszDelimiters(pszDelimiters), 
  m_bInQuote(false)
{
}

CDavStringTokenizer::~CDavStringTokenizer()
{
}

void 
CDavStringTokenizer::SetDelimiter(const char* pszDelimiters)
{
  if(pszDelimiters)
    m_pszDelimiters = pszDelimiters;
}

void 
CDavStringTokenizer::SetToken(const char* pszInput)
{
  if(pszInput){
    m_pszCurrentPos = pszInput;
    m_pszHead = pszInput;
    m_bInQuote = false;
  }
}

bool 
CDavStringTokenizer::IsDelimiter(char chInput)
{
  const char* pszDelimiter;
  for(pszDelimiter = m_pszDelimiters;
      *pszDelimiter != '\0';
      pszDelimiter++){
    if(chInput == *pszDelimiter)
      return true;
  }
  return false;
}

bool 
CDavStringTokenizer::GetNextToken(OI_STRING_A& strOutput)
{
  if(!m_pszHead || !m_pszCurrentPos)
    return false;

  bool bHasToken = false;
  strOutput.erase();

  char chCurrent;

  for(;(chCurrent = *m_pszCurrentPos) != '\0'; m_pszCurrentPos++){
    if(!m_bInQuote){
      if(IsDelimiter(chCurrent))
	if(bHasToken) 
	  return true;
	else
	  continue;
    }
    //hard code quotation as an arbitraty delimiter
    if(chCurrent == '\"'){
      m_bInQuote = !m_bInQuote;
    }
    strOutput += chCurrent;
    bHasToken = true;
  }
  return bHasToken;
}

void
MD5Hash(const void* pvData,
	unsigned long unSize,
	OI_STRING_A& output)
{
  if(!pvData || !unSize)
    return;
  unsigned char chDigest[MD5_DIGEST_LENGTH];
  MD5((const unsigned char*)pvData, unSize, chDigest);

  int nIndex;
  output.erase();
  for(nIndex = 0; nIndex < MD5_DIGEST_LENGTH; nIndex++){
    output += OI_HEX2ASC(chDigest[nIndex] >> 4);
    output += OI_HEX2ASC(chDigest[nIndex] & 0x0f);
  }
}

void
MD5HashString(const OI_STRING_A& input, OI_STRING_A& output)
{
  MD5Hash((const void*)(input.c_str()),
	  (unsigned long)(input.length()),
	  output);
}

//time parsing functions
static const char* pszRFC1123Weekdays[7] = { 
  "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" 
};

static const char* pszShortMonths[12] = { 
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};


static const char* pszRFC2459Months[12] = { 
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

//ASN1 time uses RFC2459 format
time_t 
ParseTimeRFC2459(const char* pszTime)
{
  struct tm gmt = {0};
  int n;
  char mon[4];
  n = sscanf(pszTime, OI_TIMEFORMAT_RFC2459,
	     mon, &gmt.tm_mday, 
	     &gmt.tm_hour, &gmt.tm_min, &gmt.tm_sec,
	     &gmt.tm_year);
  gmt.tm_year -= 1900;
  /* portable to check n here? */
  for (n=0; n<12; n++){
    if (strcmp(mon, pszRFC2459Months[n]) == 0)
      break;
  }
  /* tm_mon comes out as 12 if the month is corrupt, which is desired,
   * since the mktime will then fail */
  gmt.tm_mon = n;
  gmt.tm_isdst = -1;
  return mktime(&gmt);// + GMTOFF(gmt);
}

time_t 
ParseTime(const char* pszTime)
{
  time_t tmp;
  tmp = ParseTimeRFC1123(pszTime);
  if (tmp == -1) {
    tmp = ParseTimeRFC1036(pszTime);
    if (tmp == -1)
      tmp = ParseTimeASC(pszTime);
  }
  return tmp;
}

time_t 
ParseTimeASC(const char* pszTime)
{
  struct tm gmt = {0};
  int n;
  char wkday[4], mon[4];
  n = sscanf(pszTime, OI_TIMEFORMAT_ASC,
	     wkday, mon, &gmt.tm_mday, 
	     &gmt.tm_hour, &gmt.tm_min, &gmt.tm_sec,
	     &gmt.tm_year);
  /* portable to check n here? */
  for (n=0; n<12; n++)
    if (strcmp(mon, pszShortMonths[n]) == 0)
      break;
  /* tm_mon comes out as 12 if the month is corrupt, which is desired,
   * since the mktime will then fail */
  gmt.tm_mon = n;
  gmt.tm_isdst = -1;
  return mktime(&gmt);// + GMTOFF(gmt);
}

time_t 
ParseTimeISO8601(const char* pszTime)
{
  struct tm gmt = {0};
  int off_hour, off_min;
  double sec;
  off_t fix;
  int n;

  if ((n = sscanf(pszTime, OI_TIMEFORMAT_ISO8601_P,
		  &gmt.tm_year, &gmt.tm_mon, &gmt.tm_mday,
		  &gmt.tm_hour, &gmt.tm_min, &sec,
		  &off_hour, &off_min)) == 8) {
    /*  it goes: ISO8601: 2001-01-01T12:30:00+03:30 */
    gmt.tm_sec = (int)sec;
    fix = - off_hour * 3600 - off_min * 60;
  } else if ((n = sscanf(pszTime, OI_TIMEFORMAT_ISO8601_M,
		       &gmt.tm_year, &gmt.tm_mon, &gmt.tm_mday,
		       &gmt.tm_hour, &gmt.tm_min, &sec,
		       &off_hour, &off_min)) == 8) {
    /*  it goes: ISO8601: 2001-01-01T12:30:00-03:30 */
    gmt.tm_sec = (int)sec;
    fix = off_hour * 3600 + off_min * 60;
  } else if ((n = sscanf(pszTime, OI_TIMEFORMAT_ISO8601_Z,
			 &gmt.tm_year, &gmt.tm_mon, &gmt.tm_mday,
			 &gmt.tm_hour, &gmt.tm_min, &sec)) == 6) {
    /*  it goes: ISO8601: 2001-01-01T12:30:00Z */
    gmt.tm_sec = (int)sec;
    fix = 0;
  } else {
    return (time_t)-1;
  }

  gmt.tm_year -= 1900;
  gmt.tm_isdst = -1;
  gmt.tm_mon--;

  return mktime(&gmt) + fix;
}

time_t 
ParseTimeRFC1036(const char* pszTime)
{
  struct tm gmt = {0};
  int n;
  char wkday[10], mon[4];
  /* RFC850/1036 style dates: Sunday, 06-Nov-94 08:49:37 GMT */
  n = sscanf(pszTime, OI_TIMEFORMAT_RFC1036,
	     wkday, &gmt.tm_mday, mon, &gmt.tm_year,
	     &gmt.tm_hour, &gmt.tm_min, &gmt.tm_sec);
  if (n != 7) {
    return (time_t)-1;
  }

  /* portable to check n here? */
  for (n=0; n<12; n++)
    if (strcmp(mon, pszShortMonths[n]) == 0)
      break;
  /* tm_mon comes out as 12 if the month is corrupt, which is desired,
   * since the mktime will then fail */

  /* Defeat Y2K bug. */
  if (gmt.tm_year < 50)
    gmt.tm_year += 100;

  gmt.tm_mon = n;
  gmt.tm_isdst = -1;
  return mktime(&gmt);// + GMTOFF(gmt);
}

time_t 
ParseTimeRFC1123(const char* pszTime)
{
  struct tm gmt = {0};
  char wkday[4], mon[4];
  int n;
  /*  it goes: Sun, 06 Nov 1994 08:49:37 GMT */
  n = sscanf(pszTime, OI_TIMEFORMAT_RFC1123,
	     wkday, &gmt.tm_mday, mon, &gmt.tm_year, &gmt.tm_hour,
	     &gmt.tm_min, &gmt.tm_sec);
  /* Is it portable to check n==7 here? */
  gmt.tm_year -= 1900;
  for (n=0; n<12; n++)
    if (strcmp(mon, pszShortMonths[n]) == 0)
      break;
  /* tm_mon comes out as 12 if the month is corrupt, which is desired,
   * since the mktime will then fail */
  gmt.tm_mon = n;
  gmt.tm_isdst = -1;
  return mktime(&gmt);// + GMTOFF(gmt);
}

time_t 
GmtToLocal(time_t gmt)
{
#ifdef WIN32
  if (gmt == -1) 
    return -1;

  TIME_ZONE_INFORMATION tzinfo;
  DWORD dwStandardDaylight = GetTimeZoneInformation(&tzinfo);
  long bias = tzinfo.Bias;
  if (dwStandardDaylight == TIME_ZONE_ID_STANDARD) 
    bias += tzinfo.StandardBias;

  if (dwStandardDaylight == TIME_ZONE_ID_DAYLIGHT)
    bias += tzinfo.DaylightBias;

  return gmt - (bias * 60);
#else
  return gmt;	//TODO: implement this
#endif //WIN32
}

int
ReadCertificates(BIO* pBIO,
		 OI_SSL_CRTMSG enuMsgFormat,
		 OI_SSL_CRTENC enuEncFormat,
		 SSLCERTLOADCALLBACK pfnCertCB,
		 void* pvUserData)
{
  int nRet = 0;
  if(!pBIO)
    return false;

  switch(enuMsgFormat) {
  case MSG_CRT_X509:
    {
      X509* pX509 = NULL;
      switch(enuEncFormat) {
      case ENC_CRT_DER:
	pX509 = d2i_X509_bio(pBIO, NULL);
	break;
      case ENC_CRT_PEM:
	pX509 = PEM_read_bio_X509(pBIO, NULL, NULL, NULL);
	break;
      default:
	break;
      }

      if(pX509) {
	nRet = 1;
	if(pfnCertCB)
	  pfnCertCB(pX509, pvUserData);
	X509_free(pX509);
      }
    }
    break;
  case MSG_CRT_PKCS7:
    {
      PKCS7* pP7 = NULL;
      switch(enuEncFormat) {
      case ENC_CRT_DER:
	pP7 = d2i_PKCS7_bio(pBIO, NULL);
	break;
      case ENC_CRT_PEM:
	pP7 = PEM_read_bio_PKCS7(pBIO, NULL, NULL, NULL);
	break;
      default:
	break;
      }

      if(pP7) {
	STACK_OF(X509) *pCerts = NULL;
	int nID = OBJ_obj2nid(pP7->type);

	switch(nID) {
	case NID_pkcs7_signed:
	  pCerts = pP7->d.sign->cert;
	  break;
	case NID_pkcs7_signedAndEnveloped:
	  pCerts = pP7->d.signed_and_enveloped->cert;
	  break;
	default:
	  break;
	}

	if(pCerts) {
	  X509* pX509;
	  int nIndex;
	  int nCount = sk_X509_num(pCerts);
	  for(nIndex = 0; nIndex < nCount; nIndex++) {
	    pX509 = sk_X509_value(pCerts, nIndex);
	    if(pX509) {
	      nRet++;
	      if(pfnCertCB)
		pfnCertCB(pX509, pvUserData);
	      //X509_free(pX509);
	    }
	  }
	}
	PKCS7_free(pP7);
      }
    }
    break;
  default:
    break;
  }

  return nRet;
}

static void
loadCertCB(X509* pX509, void* pvUserData)
{
  OI_ASSERT(pvUserData);
  X509_STORE* pStore = (X509_STORE*)pvUserData;
  X509_STORE_add_cert(pStore, pX509); /* automatically increases ref*/
}

/*
 * Load certificates from a file into ssl cert store
 */
int
LoadCertificates(const char* pszPath,
		 OI_SSL_CRTMSG enuMsgFormat,
		 OI_SSL_CRTENC enuEncFormat)
{
  int nRet;
  if(!pszPath || !g_pSSLCtx)
    return false;
  BIO* pBIO = BIO_new(BIO_s_file());
  if(!BIO_read_filename(pBIO, pszPath))
    return false;
  X509_STORE* pStore = SSL_CTX_get_cert_store(g_pSSLCtx);
  nRet =  ReadCertificates(pBIO, enuMsgFormat, enuEncFormat,
			   loadCertCB, (void*)pStore);
  BIO_free(pBIO);
  return nRet;
}

/**
 * Load certificates from a cert list into ssl cert store
 * return value:
 *	number of cert successfully loaded
 */
int
LoadCertificates(CDavSSLCertificateList* pList)
{
  OI_ASSERT(pList);
  X509_STORE* pStore = SSL_CTX_get_cert_store(g_pSSLCtx);
  unsigned int unCount = pList->GetCount();
  unsigned int unIndex;
  int nLoaded = 0;
  for(unIndex = 0; unIndex < unCount; unIndex++) {
    X509* pX509 = (*pList)[unIndex].GetX509();
    if(pX509) {
      X509_STORE_add_cert(pStore, pX509); /* automatically increases ref*/
      nLoaded++;
    }
  }
  return nLoaded;
}

/**
 * Load a single internal cert into cert store
 * return value:
 *	1	-- successful
 *	0	-- failed
 */
int
LoadCertificate(CDavSSLCertificate* pCert)
{
  if(!pCert)
    return 0;
  X509* pX509 = pCert->GetX509();
  if(!pX509)
    return 0;
  X509_STORE* pStore = SSL_CTX_get_cert_store(g_pSSLCtx);
  if(!pStore)
    return 0;
  return X509_STORE_add_cert(pStore, pX509);
}

void
ClearCertificateStore()
{
  OI_ASSERT(g_pSSLCtx);
  X509_STORE* pStore = X509_STORE_new();
  if(pStore) {
    CRYPTO_w_lock(CRYPTO_LOCK_SSL_CTX);
    SSL_CTX_set_cert_store(g_pSSLCtx, pStore);
    CRYPTO_w_unlock(CRYPTO_LOCK_SSL_CTX);
  }
}

bool
LoadClientCertificate(CDavSSLClientCertificate* pCert)
{
  if(!pCert)
    return false;
  if(!pCert->IsValid())
    return false;
  X509* pX509 = pCert->Cert().GetX509();
  EVP_PKEY* pKey = pCert->Key();

  if(!SSL_CTX_use_certificate(g_pSSLCtx, pX509))
    return false;
  if(!SSL_CTX_use_PrivateKey(g_pSSLCtx, pKey))
    return false;
  return true;
}
